#ifndef _Work_Work_h
#define _Work_Work_h

#include <Core/Core.h>

namespace Upp {

class Work;

struct  _WorkEntry {
    enum    Status      { WORKING, FINISHED, FAILED };
    Mutex               lock;
    Semaphore           semaphore;
    std::exception_ptr  exc;
    Tuple<int, String>  error;
    volatile Atomic     status;

    inline void         SetExc(std::exception_ptr&& e)  { Mutex::Lock __(lock); exc = pick(e); }
    inline int          GetError()                      { Mutex::Lock __(lock); return error.Get<int>(); }
    inline String       GetErrorDesc()                  { Mutex::Lock __(lock); return error.Get<String>(); }
    inline void         Halt(int rc, const String& msg) { status = FAILED; error = MakeTuple<int, String>(rc, msg); semaphore.Release(); }
    inline void         Wait()                          { if(status == WORKING) semaphore.Wait(); }
    inline void         Finish()                        { status = FINISHED; semaphore.Release(); }
    _WorkEntry()        { status = WORKING; exc = NULL; }
};

template<class T> struct WorkEntry : _WorkEntry {
    T                   data;
    template<class F, class... Args> void Set(F&& fn, Args&&... args) { data = pick(fn(args...));}
};
template<> struct WorkEntry<void> : _WorkEntry {
    template<class F, class... Args> void Set(F&& fn, Args&&... args) { fn(args...);}
};

template<class T> class _WorkResult : Moveable<Pte<_WorkResult<T>>> {
protected:
    friend class        Work;
    WorkEntry<T>        *entry;
    void                Pick(_WorkResult&& wr)  { entry = pick(wr.entry); wr.entry = NULL; }
    void                Rethrow()               { ASSERT(entry); Mutex::Lock __(entry->lock); if(entry->exc) std::rethrow_exception(entry->exc); }

public:
    bool                IsFinished() const      { ASSERT(entry); return !entry->status != _WorkEntry::WORKING; }
    bool                IsError()               { Rethrow(); return  entry->status == _WorkEntry::FAILED;  }
    int                 GetError()              { ASSERT(entry); return entry->GetError(); }
    String              GetErrorDesc()          { ASSERT(entry); return entry->GetErrorDesc(); }
    void                operator=(_WorkEntry& e){ entry = & e; }
    _WorkResult()                               { entry = NULL; }
    _WorkResult(_WorkResult&& wr)               { Pick(pick(wr)); }
    void operator=(_WorkResult&& wr)            { Pick(pick(wr)); }
    virtual ~_WorkResult()                      { if(entry) { entry->Wait(); delete entry; }}
};

template<class T> struct WorkResult : _WorkResult<T> {
    T&                  Get()                   { _WorkResult<T>::Rethrow(); _WorkResult<T>::entry->Wait(); return _WorkResult<T>::entry->data; }
    const T&            operator~()             { return Get(); }
};

template<> struct WorkResult<void> :  _WorkResult<void> {
    void                Get()                   { _WorkResult<void>::Rethrow(); _WorkResult<void>::entry->Wait(); return void(); }
};

class Work {
private:
    CoWork cowork;

public:

    struct Error : Exc {
        int code;
        Error() : code(-1) {}
        Error(const String& reason) : code(-1), Exc(reason) {}
        Error(int rc, const String& reason) : code(rc), Exc(reason) {};
    };

    template< class Function, class... Args>
    WorkResult<std::result_of_t<std::decay_t<Function>(std::decay_t<Args>...)>>
    Do(Function&& f, Args&&... args)
    {
        auto *entry = new WorkEntry<std::result_of_t<std::decay_t<Function>(std::decay_t<Args>...)>>();
        WorkResult<std::result_of_t<std::decay_t<Function>(std::decay_t<Args>...)>> workresult;
        workresult.entry = entry;

        cowork & [=, entry = pick(entry)]() mutable {
            try {
                entry->Set(f);
                entry->Finish();
            }
            catch(Work::Error& e) {
                entry->Halt(e.code, e);
                RLOG("Worker failed. " << e);
            }
            catch(...) {
                entry->SetExc(std::current_exception());
                entry->Halt(-1, "Exception raised.");
                RLOG("Exception raised.");
            }
        };
        return pick(workresult);
    }
    inline void Finish() { cowork.Finish(); }
};
}
#endif
