#include "OCE.h"

#ifdef flagGTK

#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <gdk/gdkprivate.h>

namespace Upp {

XIM GtkDHCtrl::xim = 0;

static Atom XAtomRaw(const char *name)
{
	return XInternAtom(GetX11Display(), name, 0);
}

static Atom XAtom(const char *name)
{
	GuiLock __; 
	Atom x;
	INTERLOCKED {
		static VectorMap<String, int> atoms;
		int q = atoms.Get(name, Null);
		if(IsNull(q)) {
			q = XAtomRaw(name);
			atoms.Add(name, q);
		}
		x = q;
	}
	return x;
}

/////////////////////////////////////////////////////////////////////////////////////////
// Constructor
GtkDHCtrl::GtkDHCtrl()
{
	// Sets control NOT initialized
	isInitialized = false;

	// Sets control NOT mapped
	isMapped = false;

	// Resets error contition
	isError = false;

	// Sets the user visual to null
	UserVisualInfo = 0;

	// No background painting
	NoBackPaint();

	hwnd = 0;
	
} // END Constructor class GtkDHCtrl

/////////////////////////////////////////////////////////////////////////////////////////
// Destructor
GtkDHCtrl::~GtkDHCtrl()
{
	// Destroys the associated window and clean up stuffs
	Terminate();

} // END Destructor class GtkDHCtrl

ArrayMap<Window, GtkDHCtrl::XWindow>& GtkDHCtrl::Xwindow()
{
	return Single< ArrayMap<Window, XWindow> >();
}

GtkDHCtrl::XWindow *GtkDHCtrl::AddXWindow(Window &w)
{
	GuiLock __;
	int i = Xwindow().Find(None);
	if(i >= 0)
		Xwindow().SetKey(i, w);
	XWindow& cw = i >= 0 ? Xwindow()[i] : Xwindow().Add(w);
	cw.ctrl    = this;
	cw.exposed = true;
	cw.owner   = GetParent();
	cw.xic     = NULL;
	return &cw;
}

void GtkDHCtrl::RemoveXWindow(Window &w)
{
	GuiLock __;
	int i = Xwindow().Find(w);
	if(i >= 0) {
		Xwindow().SetKey(i, None);
		Xwindow()[i].ctrl = NULL;
	}

}

GtkDHCtrl::XWindow *GtkDHCtrl::XWindowFromWindow(Window &w)
{
	GuiLock __;
	int i = Xwindow().Find(w);
	if(i >= 0)
		return &Xwindow()[i];
	else
		return NULL;
}

Rect GtkDHCtrl::GetRectInParentWindow(void)
{
	GuiLock __;
	Rect r = GetScreenRect();
	Ctrl *q = GetParent();
	while(q && q->GetParent())
		q = q->GetParent();
	if(q)
	{
		Rect pr = q->GetScreenRect();
		r -= pr.TopLeft();
	}
	return r;
}


/////////////////////////////////////////////////////////////////////////////////////////
// Get parent Window (the one in containint TopCtrl
Window GtkDHCtrl::GetParentWindow(void)
{
	return GDK_WINDOW_XWINDOW(GetTopWindow()->gdk());
}

/////////////////////////////////////////////////////////////////////////////////////////
// Get current X Display
XDisplay *GtkDHCtrl::GetXDisplay(void)
{
	static XDisplay *xd = NULL;
	
	if(!xd)
	{
		GtkWidget *w = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		GdkDisplay *d = gtk_widget_get_display(w);
		xd = gdk_x11_display_get_xdisplay(d);
		gtk_widget_destroy(w);
	}
	return xd;
}

/////////////////////////////////////////////////////////////////////////////////////////
// Maps/unmaps the window
void GtkDHCtrl::MapWindow(bool map)
{
	GuiLock __; 
	
	// no action if not initialized
	if(!isInitialized)
		return;

	if(map && !isMapped)
		XMapWindow(GetXDisplay(), hwnd);
	else if(!map && isMapped)
		XUnmapWindow(GetXDisplay(), hwnd);

	isMapped = map;

} // END GtkDHCtrl::MapWndow()


/////////////////////////////////////////////////////////////////////////////////////////
// Initializes the view
bool GtkDHCtrl::Init()
{
	GuiLock __; 
	static bool isInitializing = false;

	// Just for security sakes...
	if(isInitialized)
		return true;

	// Prevents reentrant call....
	if(isInitializing)
		return false;
	isInitializing = true;

	// Call BeforeInit user func...
	BeforeInit();

	// if display is null, error
	if(!GetXDisplay())
	{
		// Call AfterInit user func...
		AfterInit(true);

		// Sets the appropriate error message
		SetErrorMessage("GTKDHCtrl : Bad display");

		isError = true;
		isInitializing = false;
		return false;
	}

	// Calls the user visual function
	UserVisualInfo = CreateVisual();

	// If error, returns
	if(isError)
	{
		isInitializing = false;
		return false;
	}

	// Gets the default visual, if none is given
	Visual *visual;
	int Depth;
	if(UserVisualInfo)
	{
		visual = UserVisualInfo->visual;
		Depth = UserVisualInfo->depth;
	}
	else
	{
		visual = DefaultVisual(GetXDisplay(), DefaultScreen(GetXDisplay()));
		Depth = DefaultDepth(GetXDisplay(), DefaultScreen(GetXDisplay()));
	}

	// Initializes attribute setting flags
	unsigned long ValueMask =
		CWBorderPixel
		| CWColormap
		| CWSaveUnder
		| CWColormap
	; // END ValueMask

	// Initializes attribute structure
	XSetWindowAttributes winAttributes;
	// creates a ColorMap, in case we're not using default visual
	winAttributes.colormap = XCreateColormap(GetXDisplay(), GetParentWindow(), visual, AllocNone);
	winAttributes.border_pixel = 0;
	winAttributes.save_under = 0; // XFalse

	// Calls the attributes user setting routine
	SetAttributes(ValueMask, winAttributes);
	
	// get the parent window
	Window parentWindow = GetParentWindow();

	// Creates the X11 window
	Rect r = GetRectInParentWindow();
	if (!hwnd) {
		hwnd = XCreateWindow
		(
			GetXDisplay(),								// display
	//		GetTopWindow()->GetWindow(), 				// parent
			parentWindow,
	
			r.left, r.top, r.Width(), r.Height(),		// x, y, width, height
			0,											// border width
			Depth,										// depth
			InputOutput,								// class
			visual,										// visual
			ValueMask,									// value mask
			&winAttributes								// attributes
		);
	} else {
		XReparentWindow(GetXDisplay(), hwnd, GetParentWindow(), r.left, r.top);
		XResizeWindow(GetXDisplay(), hwnd, r.Width(), r.Height());
		XChangeWindowAttributes(GetXDisplay(), hwnd, ValueMask, &winAttributes);
	}

	// Frees VisualInfo
    if (UserVisualInfo)
    {
        XFree( (char *)UserVisualInfo);
        UserVisualInfo = 0;
    }

    // If problem creating window, error
    if(!hwnd)
    {
		// Call AfterInit user func...
		AfterInit(true);

		// Sets the appropriate error message
		SetErrorMessage("DHCtrl : Can't create window");

        isError = true;
		isInitializing = false;
        return false;
    }

	// Adds window to UPP managed windows
	XWindow *cw = AddXWindow(hwnd);

	cw->xic = xim ? XCreateIC
					(
						xim,
						XNInputStyle,
						XIMPreeditNothing | XIMStatusNothing,
						XNClientWindow,
						hwnd,
						XNFocusWindow,
						hwnd,
	    				NULL
	    			)
	    : NULL;

/* DO NOT HANDLE INPUTS -- IT'LL BE DONE BY PARENT WINDOW
	long im_event_mask = 0;
	if(cw->xic)
		XGetICValues(cw->xic, XNFilterEvents, &im_event_mask, NULL);

	XSelectInput
	(
		GetXDisplay(),
		hwnd,
		ExposureMask
//		| StructureNotifyMask		// *very* important, flag MUST NOT be set
		| KeyPressMask
		| FocusChangeMask
		| KeyPressMask
		| KeyReleaseMask
		| PointerMotionMask
		| ButtonPressMask
		| ButtonReleaseMask
		| PropertyChangeMask
		| VisibilityChangeMask
		| im_event_mask
	);
*/

	int version = 5;
	XChangeProperty
	(
		GetXDisplay(),
		hwnd,
		XAtom("XdndAware"),
		XA_ATOM,
		32,
		0,
		(byte *)&version,
		1
	);

	// Maps the window if needed
	if(IsShown())
		MapWindow(true);
	
	// Flushes the display
    XFlush(GetXDisplay());

    // Stores the initial control size
    CurrentSize = GetSize();

	// Exits from initializing lock
	isInitializing = false;

	// mark control as initialized
	isInitialized = true;

	// Resets the message
	isError = false;
	SetErrorMessage("");

	// Call AfterInit user func...
	AfterInit(false);

	return true;

} // END GtkDHCtrl::Init()


/////////////////////////////////////////////////////////////////////////////////////////
// Terminates the view
void GtkDHCtrl::Terminate(void)
{
	GuiLock __; 
	BeforeTerminate();

	if(!isInitialized)
		return;

	// Unmaps the window
	MapWindow(false);

	// gathers data from XWindow (needs Input Context...)
	XWindow *cw = XWindowFromWindow(hwnd);

	// Frees input context as needed
	if(cw->xic)
	{
		XDestroyIC(cw->xic);
		cw->xic = NULL;
	}

	// Removes XWindow from Upp list
	RemoveXWindow(hwnd);

	// Destroys the window
	// Not to do, it's done destroying the parent window by X11 system
//	XDestroyWindow(GetXDisplay(), hwnd);

	// Resets initialization and error flags
	isInitialized = false;
	isError = false;

} // END GtkDHCtrl::Terminate()

/////////////////////////////////////////////////////////////////////////////////////////
// State handler
void GtkDHCtrl::State(int reason)
{
	GuiLock __; 

	Rect r;

	// No handling if in error state
	if( isError)
		return;

	// Initializes the control if needed (and possible...)
	if(!isInitialized && GetTopWindow())
		Init();

	if(isInitialized)
	{
		switch( reason )
		{
			case FOCUS      : // = 10,
				break;

			case ACTIVATE   : // = 11,
				break;

			case DEACTIVATE : // = 12,
				break;

			case SHOW       : // = 13,
				MapWindow(IsVisible());
				break;

			case ENABLE     : // = 14,
				break;

			case EDITABLE   : // = 15,
				break;

			case OPEN       : // = 16,
				MapWindow(IsShown());
				break;

			case CLOSE      : // = 17,
				Terminate();
				break;

			case POSITION   : // = 100,
			case LAYOUTPOS  : // = 101,
				r = GetRectInParentWindow();
				XMoveResizeWindow(GetXDisplay(), hwnd, r.left, r.top, r.Width(), r.Height());
				break;

			default:
				break;

		} // switch(reason)
	}
} // END GtkDHCtrl::State()

/////////////////////////////////////////////////////////////////////////////////////////
// Property Visual
Visual *GtkDHCtrl::GetVisual(void)
{
	GuiLock __; 
	if(UserVisualInfo)
		return UserVisualInfo->visual;
	else
		return DefaultVisual(GetXDisplay(), DefaultScreen(GetXDisplay()));

} // END GtkDHCtrl::getVisual()


/////////////////////////////////////////////////////////////////////////////////////////
// Property VisualInfo
XVisualInfo GtkDHCtrl::GetVisualInfo(void)
{
	GuiLock __; 
	// if present an user visual info, just return it
	if(UserVisualInfo)
		return *UserVisualInfo;

	XVisualInfo visualInfo;
	memset(&visualInfo, 0, sizeof(visualInfo));

	// get the active visual
	Visual *visual = GetVisual();

	// gets a list of all available XVisualinfo
	XVisualInfo *v = 0;
	XVisualInfo vtemplate;
	int nVis;
	XVisualInfo *vlist = XGetVisualInfo(GetXDisplay(), VisualNoMask, &vtemplate, &nVis);

	// search for current visual inside the list
	if(vlist)
	{
		for (v = vlist; v < vlist + nVis; v++)
		{
			if (v->visual == visual)
			{
				visualInfo = *v;
				break;
			}
		}
	    XFree(vlist);
	}
	else
	{
		isError = true;
		ErrorMessage = "DHCtrl: no XVisualInfo for current Visual";
	}

	// returns the found XVisualInfo struct
	return visualInfo;

} // END GtkDHCtrl::GetVisualInfo()

}

#endif