#ifndef _PlantUmlEditor_SyntaxParser_h_
#define _PlantUmlEditor_SyntaxParser_h_


#include <stdexcept>
#include <string>
#include <sstream>


namespace PlantUML
{
	// ======================================================================
	//
	// ======================================================================
	#define PUML_HL_StyleDesc(name, ink, paper, shortDesc, desc)      name,
	typedef enum HighlighType {
			#include "hl_color.i"
		PUML_HL_COUNT
		} HighlighType;
	#undef PUML_HL_StyleDesc


	// ======================================================================
	//
	// ======================================================================
	#define PUML_HL_StyleDesc(name, ink, paper, shortDesc, desc)      #name,
	template<class T>
	T& operator<<(T& str, const HighlighType hlt) {
		static const char* PUML_HL_TypeNameString[PUML_HL_COUNT] = {
			#include "hl_color.i"
			};
	#undef PUML_HL_StyleDesc
			
			if (hlt < PUML_HL_COUNT) {
				str << PUML_HL_TypeNameString[hlt];
			}
			else
			{
				str <<"UNKNOWN__HIGHLIGHT__VAL[ " << int(hlt) << " ]";
			}
			return str;
	}



	// ======================================================================
	//
	// ======================================================================
	class PlantUmlParseException : public std::runtime_error
	{
		public:
			PlantUmlParseException(const std::string& msg)
				: std::runtime_error(msg)
				{};
			
			PlantUmlParseException(const std::string& msg, exception& cause)
				: std::runtime_error( msg + "\n Caused by :   " + cause.what())
				{};
			
			virtual ~PlantUmlParseException() throw() {};
	};


	// ======================================================================
	//
	// ======================================================================
	typedef enum PlantUMLParsingType {
		 PUML_PARSE_ALL = 0
		,PUML_PARSE_NOTE_TEXT
		,PUML_PARSE_REF_TEXT
		,PUML_PARSE_LEGEND_TEXT
		,PUML_PARSE_SECTION_TEXT
		,PUML_PARSE_BOX
		,PUML_PARSE_SKIN_PARAMS
		
		, PUML_PARSE_COUNT
	} PlantUMLParsingType;


	static const char* PlantUMLParsingType_nameString[PUML_PARSE_COUNT] = {
			"PUML_PARSE_ALL",
			"PUML_PARSE_NOTE_TEXT",
			"PUML_PARSE_REF_TEXT",
			"PUML_PARSE_LEGEND_TEXT",
			"PUML_PARSE_SECTION_TEXT",
			"PUML_PARSE_BOX",
			"PUML_PARSE_SKIN_PARAMS",
		};

#ifndef CORE_H
	template<class T>
	T& operator<<(T& str, const PlantUMLParsingType p) {
			
			if (p < PUML_PARSE_COUNT) {
				str << PlantUMLParsingType_nameString[p];
			}
			else
			{
				str <<"UNKNOWN__PARSING_TYPE__VAL[ " << int(p) << " ]";
			}
			return str;
	}
#endif


	typedef enum LineType {
		LINE_NORMAL,
		LINE_NOTE_START,
		LINE_NOTE_END,
		LINE_GROUP_IF_START,
		LINE_GROUP_IF_ELSE,
		LINE_GROUP_START,
		LINE_GROUP_END,
		LINE_SECTION_START,
		LINE_SECTION_END,
		LINE_LEGEND_START,
		LINE_LEGEND_END,
		LINE_BOX_START,
		LINE_BOX_END,
		LINE_REF_START,            // "ref over Bob"
		LINE_REF_END,              // "end ref"
		LINE_SKINPARAM_SEQ_START,  // "skinparam sequence {"
		LINE_SKINPARAM_SEQ_END,    // "}"
		
		LINE_TYPE_COUNT
	} LineType;

	static const char* LineType_nameString[LINE_TYPE_COUNT] = {
		"LINE_NORMAL",
		"LINE_NOTE_START",
		"LINE_NOTE_END",
		"LINE_GROUP_IF_START",
		"LINE_GROUP_IF_ELSE",
		"LINE_GROUP_START",
		"LINE_GROUP_END",
		"LINE_SECTION_START",
		"LINE_SECTION_END",
		"LINE_LEGEND_START",
		"LINE_LEGEND_END",
		"LINE_BOX_START",
		"LINE_BOX_END",
		"LINE_REF_START",            // "ref over Bob"
		"LINE_REF_END",              // "end ref"
		"LINE_SKINPARAM_SEQ_START",  // "skinparam sequence {"
		"LINE_SKINPARAM_SEQ_END",    // "}"
	};

#ifndef CORE_H
	template<class T>
	T& operator<<(T& str, const LineType p) {
			
			if (p < LINE_TYPE_COUNT) {
				str << LineType_nameString[p];
			}
			else
			{
				str <<"UNKNOWN__LINE_TYPE__VAL[ " << int(p) << " ]";
			}
			return str;
	}
#endif
};

#ifdef CORE_H
#endif

namespace PlantUML
{
	
	class LineContext {
		
		public:
		int indentLevel;
		
		private:
		LineType lineType;
		
		public:
		bool applyBlockColor;        // context of the line
		HighlighType blockHighLight;
		PlantUMLParsingType parsingType;      // parsing type of current context
		
		
		LineContext() : indentLevel(0)
		              , lineType(LINE_NORMAL)
		              , applyBlockColor(false)
		              , blockHighLight(PUML_STYLE_NORMAL)
		              , parsingType(PUML_PARSE_ALL) {}

		LineContext(const LineContext& e)
		              : indentLevel(e.indentLevel)
		              , lineType(e.lineType)
		              , applyBlockColor(e.applyBlockColor)
		              , blockHighLight(e.blockHighLight)
		              , parsingType(e.parsingType) {}

		~LineContext() {}
		
		bool updateLineType(LineType lt) {
			bool res = (lt != lineType);
			lineType=lt;
			return res;
		}
		
		inline LineType getLineType() const { return lineType; }
		
		PlantUMLParsingType getNextParsingType() const; // parsingType for the next line
		bool IndentNextLine() const; // Indent next line needed
		bool UnIndentLine() const;   // Un indent of current line needed
		              
	};
	
	class LineParserActions {
		public:
		virtual ~LineParserActions() {};
		
		// COLORISING METHODS
		virtual void colorizeText(const std::wstring::const_iterator& start, const std::wstring::const_iterator& end, HighlighType hlt) = 0;
		virtual void setLineType(LineType l) = 0;

		// SYNTAX LOADING METHODS  (Macro names, ....)
		virtual bool isSymbolsAdd() const = 0;

	};

	
	// ======================================================================
	//
	// ======================================================================
	// Class Declaration
	class PlantUMLParser
	{
		private:
			PlantUMLParser() {};
			virtual ~PlantUMLParser() {};

		public:
			static bool parseLine(const std::wstring& inputText, LineParserActions& hlp, PlantUMLParsingType pt) throw (PlantUmlParseException);
			static void clearSymbols();
	};

	void logFct(const std::wstringstream& str);
	void logFct(const std::stringstream& str);
	void logFct(const std::wstring& str);
	void logFct(const std::string& str);
	void logFct(const char* str);
	void logFct(const wchar_t* str);

}; // namespace PlantUML



#endif
