#include <CodeEditor/CodeEditor.h>


#include "PlantUmlEditor.h"
#include "LineParserActions.h"




#define PUML_HL_StyleDesc(name, ink, paper, shortDesc, desc)      ink,
static const int puml_default_style_ink[PlantUML::PUML_HL_COUNT] = {
		#include "hl_color.i"
	};
#undef PUML_HL_StyleDesc


#define PUML_HL_StyleDesc(name, ink, paper, shortDesc, desc)      paper,
static const int puml_default_style_paper[PlantUML::PUML_HL_COUNT] = {
		#include "hl_color.i"
	};
#undef PUML_HL_StyleDesc






// ===================================================================
//
// ===================================================================
namespace PlantUML {
	void logFct(const char* str) {
		LOG(str);
	}
	
	void logFct(const wchar_t* str) {
		LOG(str);
	}

	void logFct(const std::wstringstream& str) {
		Upp::WString s(str.str());
		LOG(s);
	}

	void logFct(const std::stringstream& str) {
		Upp::String s(str.str());
		LOG(s);
	}
	
	void logFct(const std::wstring& str) {
		Upp::WString s(str);
		LOG(s);
	}
	
	void logFct(const std::string& str) {
		Upp::String s(str);
		LOG(s);
	}
};






// ===================================================================
//
// ===================================================================
namespace PlantUML {
	bool LineContext::IndentNextLine() const { // parsingType for the next line
		if (  (lineType == LINE_NOTE_START)
			||(lineType == LINE_GROUP_IF_START)
			||(lineType == LINE_GROUP_IF_ELSE)
			||(lineType == LINE_GROUP_START)
			||(lineType == LINE_SECTION_START)
			||(lineType == LINE_REF_START)
			||(lineType == LINE_LEGEND_START)
			||(lineType == LINE_BOX_START)
			||(lineType == LINE_SKINPARAM_SEQ_START) ) {
			return true;
		}
		else {
			return false;
		}
	}

	bool LineContext::UnIndentLine() const { // parsingType for the next line
		if (  (lineType == LINE_NOTE_END)
			||(lineType == LINE_GROUP_END)
			||(lineType == LINE_SECTION_END)
			||(lineType == LINE_LEGEND_END)
			||(lineType == LINE_BOX_END)
			||(lineType == LINE_REF_END)
			||(lineType == LINE_SKINPARAM_SEQ_END) ){
			return true;
		}
		else {
			return false;
		}
	}



	PlantUMLParsingType LineContext::getNextParsingType() const { // parsingType for the next line
		// TODO   replace with CONVERT TABLE ARRAY
		switch (parsingType) {
			case PUML_PARSE_ALL:
				if (lineType == LINE_NOTE_START)           return PUML_PARSE_NOTE_TEXT;
				if (lineType == LINE_REF_START)            return PUML_PARSE_REF_TEXT;
				if (lineType == LINE_LEGEND_START)         return PUML_PARSE_LEGEND_TEXT;
				if (lineType == LINE_SECTION_START)        return PUML_PARSE_SECTION_TEXT;
				if (lineType == LINE_BOX_START)            return PUML_PARSE_BOX;
				if (lineType == LINE_SKINPARAM_SEQ_START)  return PUML_PARSE_SKIN_PARAMS;
				break;

			case PUML_PARSE_NOTE_TEXT:
				if (lineType != LINE_NOTE_END) return PUML_PARSE_NOTE_TEXT;
				break;

			case PUML_PARSE_REF_TEXT:
				if (lineType != LINE_REF_END) return PUML_PARSE_REF_TEXT;
				break;

			case PUML_PARSE_LEGEND_TEXT:
				if (lineType != LINE_LEGEND_END) return PUML_PARSE_LEGEND_TEXT;
				break;

			case PUML_PARSE_SECTION_TEXT:
				if (lineType != LINE_SECTION_END) return PUML_PARSE_SECTION_TEXT;
				break;

			case PUML_PARSE_BOX:
				if (lineType != LINE_BOX_END) return PUML_PARSE_BOX;
				break;
				
			case PUML_PARSE_SKIN_PARAMS:
				if (lineType != LINE_SKINPARAM_SEQ_END) return PUML_PARSE_SKIN_PARAMS;
				break;
				
				
		}
		return PUML_PARSE_ALL;
	}
};

// ===================================================================
//   CLASS :  PlantUmlSyntax
// ===================================================================
Upp::Vector<PlantUML::LineContext>  PlantUmlSyntax::_lineContexts;

HlStyle   PlantUmlSyntax::puml_style_ink[ PlantUML::PUML_HL_COUNT ];
HlStyle   PlantUmlSyntax::puml_style_paper[ PlantUML::PUML_HL_COUNT ];

#define PUML_HL_StyleDesc(name, ink, paper, shortDesc, desc)      #name ,
const char* PlantUmlSyntax::puml_style_name[PlantUML::PUML_HL_COUNT] = {
		#include "hl_color.i"
	};
#undef PUML_HL_StyleDesc

#define PUML_HL_StyleDesc(name, ink, paper, shortDesc, desc)      shortDesc ,
const char* PlantUmlSyntax::puml_style_short_desc[PlantUML::PUML_HL_COUNT] = {
		#include "hl_color.i"
	};
#undef PUML_HL_StyleDesc

#define PUML_HL_StyleDesc(name, ink, paper, shortDesc, desc)      desc ,
const char* PlantUmlSyntax::puml_style_long_desc[PlantUML::PUML_HL_COUNT] = {
		#include "hl_color.i"
	};
#undef PUML_HL_StyleDesc



// ===================================================================
PlantUmlSyntax::PlantUmlSyntax()
{
}


// ===================================================================
void PlantUmlSyntax::loadDefaultSettings() {
	for (int c=0; c<PlantUML::PUML_HL_COUNT; ++c) {
		if (puml_default_style_ink[c] >=0)    puml_style_ink[c]   = HighlightSetup::hl_style[puml_default_style_ink[c]];
		else                                  puml_style_ink[c]   = HighlightSetup::hl_style[HighlightSetup::INK_NORMAL];
		
		
		if (puml_default_style_paper[c] >=0)  puml_style_paper[c] = HighlightSetup::hl_style[puml_default_style_paper[c]];
		else                                  puml_style_paper[c] = HighlightSetup::hl_style[HighlightSetup::PAPER_NORMAL];
	}
	puml_style_ink[PlantUML::PUML_STYLE_ARROW].bold = false;
	puml_style_ink[PlantUML::PUML_STYLE_PARTICIPANT].color = Color(178, 40, 141);
	puml_style_ink[PlantUML::PUML_STYLE_PARTICIPANT].italic = true;
	puml_style_ink[PlantUML::PUML_STYLE_PARTICIPANT].bold = false;
	
	puml_style_ink[PlantUML::PUML_STYLE_MSG_TEXT].color = Color(0, 0, 0);
	puml_style_paper[PlantUML::PUML_STYLE_MSG_TEXT].color = Color(220, 255, 255);//Color(214, 214, 255);
	
	puml_style_ink[PlantUML::PUML_STYLE_NOTE_TEXT].color = Color(89, 89, 89);
	puml_style_paper[PlantUML::PUML_STYLE_NOTE_TEXT].color = Color(255, 255, 183);
	
	//puml_style_paper[PlantUML::PUML_STYLE_ERROR].color = Color(255, 227, 227);

	puml_style_ink[PlantUML::PUML_STYLE_MACRO_NAME].color = Color(0, 0, 0);
	puml_style_ink[PlantUML::PUML_STYLE_MACRO_NAME].bold = true;
	
	
	
}


// ===================================================================
void PlantUmlSyntax::ScanSyntax(const wchar *start, const wchar *end, int line, int tab_size)
{
	String blockString;
	Upp::WString ustr(start,end);
	blockString << "PlantUmlSyntax::ScanSyntax( " << ustr << " )";
	LOGBLOCK(blockString);
	std::wstring str(ustr.ToStd());
	
	if (_lineContexts.GetCount() < line) {
		PlantUML::LineContext defaultLine;
		if ( !_lineContexts.IsEmpty() ) defaultLine.parsingType = _lineContexts.Top().getNextParsingType(); // set parsingType for added lines
		_lineContexts.SetCountR( line,  defaultLine );
	}
	
	PlantUML::LineParserSyntaxLoader syntaxLoader(str, line, _lineContexts);
	try {
		LOG(_lineContexts[line].parsingType);
		if ( ! PlantUML::PlantUMLParser::parseLine(str, syntaxLoader, _lineContexts[line].parsingType ) )
		{
			// TODO
		}
		else
		{
			RLOG("============= ERROR 1 ===============");
		}
	}
	catch(...) {
		RLOG("============= ERROR 2 ===============");
	}
}


// ===================================================================
void PlantUmlSyntax::Highlight(const wchar *start, const wchar *end, HighlightOutput& hls, CodeEditor *editor, int line, int64 pos)
{
	/*
	String blockString;
	blockString << "PlantUmlSyntax::Highlight( line:" << line << "  ,  pos:" << pos << " )  " <<  WString(start, end).ToString() ;
	RLOGBLOCK(blockString);
	*/
	Upp::WString ustr(start,end);
	std::wstring str(ustr.ToStd());
	
	if (_lineContexts.GetCount() < editor->GetLineCount()) {
		PlantUML::LineContext defaultLine;
		if ( !_lineContexts.IsEmpty() ) defaultLine.parsingType = _lineContexts.Top().getNextParsingType(); // set parsingType for added lines
		_lineContexts.SetCountR( editor->GetLineCount(),  defaultLine );
	}
	
	PlantUML::LineParserColorize coloriser(hls, str, line, _lineContexts, *editor);
	try {
		LOG(_lineContexts[line].parsingType);
		if ( ! PlantUML::PlantUMLParser::parseLine(str, coloriser, _lineContexts[line].parsingType ) )
		{
			coloriser.highLightRestOfLine();
		}
		else
		{
			RLOG("============= HIGHLIGHT WARNING ===============");
			coloriser.highLightWarning();
		}
	}
	catch(...) {
		RLOG("============= HIGHLIGHT ERROR ===============");
		coloriser.highLightError();
	}
}

// ===================================================================
void PlantUmlSyntax::IndentInsert(CodeEditor& editor, int chr, int count)
{
	LOGBLOCK("PlantUmlSyntax::IndentInsert");
	LOG( String("PlantUmlSyntax::IndentInsert( count=") << count << " )");
	if(chr == '\n') {
		LOG("PlantUmlSyntax::IndentInsert  === NEWLINE ===");
		while(count--) {
			const int currLine = editor.GetCursorLine();
			WString cursorLine = editor.GetWLine(currLine);
			
			editor.InsertChar('\n', 1);
			
			Identation::Type idType = FindIdentationType(editor, cursorLine);
			char idChar = GetIdentationByType(idType);

			if ( _lineContexts[currLine].IndentNextLine() ) {
				int mult = 1;
				if(idType == Identation::Space) mult = CalculateSpaceIndetationSize(editor);
				editor.InsertChar(idChar, mult);
			}
			editor.InsertChar(idChar, CalculateLineIndetations(cursorLine, idType));

			 // insert new lineContext
			_lineContexts.Insert(currLine+1).parsingType = _lineContexts[currLine].getNextParsingType(); // copy parsingType of last line
		}
	}
	if(count > 0) {
		LOG("PlantUmlSyntax::IndentInsert  === CHR ===");
		editor.InsertChar(chr, count);
	}
}

// ===================================================================
int PlantUmlSyntax::CalculateLineIndetations(const WString& line, Identation::Type type)
{
	int count = 0;
	for(int i = 0; i < line.GetLength(); i++) {
		if(type == Identation::Tab && line[i] == '\t')
			count++;
		else
		if(type == Identation::Space && line[i] == ' ')
			count++;
		else
			break;
	}
	return count;
}

// ===================================================================
PlantUmlSyntax::Identation::Type PlantUmlSyntax::FindIdentationType(CodeEditor& editor, const WString& line)
{
	Identation::Type type = Identation::Unknown;
	if(line.StartsWith("\t"))
		type = Identation::Tab;
	else
	if(line.StartsWith(" "))
		type = Identation::Space;
	else {
		for(int i = 0; i < editor.GetLineCount(); i++) {
			WString cLine = editor.GetWLine(i);
			if(cLine.StartsWith("\t")) {
				type = Identation::Tab;
				break;
			}
			else
			if(cLine.StartsWith(" ")) {
				type = Identation::Space;
				break;
			}
		}
	}
	return type;
}

// ===================================================================
int PlantUmlSyntax::CalculateSpaceIndetationSize(CodeEditor& editor)
{
	int current = 0;
	for(int i = 0; i < editor.GetLineCount(); i++) {
		WString line = editor.GetWLine(i);
		for(int j = 0; j < line.GetLength(); j++) {
			if(line[j] == ' ')
				current++;
			else
				break;
		}
		
		if(current > 0)
			break;
	}
	
	// TODO: 4 is magic numer - try to find the way to get this number from ide constants
	return current > 0 ? current : 2;
}

// ===================================================================
char PlantUmlSyntax::GetIdentationByType(Identation::Type type)
{
	if(type == Identation::Space)
		return ' ';
	return '\t';
}



// puml_style_ink[PlantUML::PUML_STYLE_MACRO_NAME].

using Upp::CParser;
using Upp::String;
using Upp::Color;

void PlantUmlSyntax::LoadColorSettings(const char *s)
{
	CParser p(s);
	p.SkipComments(true);
	
	try {
		while(!p.IsEof()) {
//			while ( p.Id("/") ) ) {
//				p.ReadLine();
//			}
			
			HlStyle* currInk = 0;
			HlStyle* currPaper = 0;
			String id = p.ReadId();

			bool bold = false;
			bool italic = false;
			bool underline = false;
			Color inkColor = ReadColor(p);
			Color paperColor = ReadColor(p);
			for(;;)
				if(p.Id("bold"))           bold = true;
				else if(p.Id("italic"))    italic = true;
				else if(p.Id("underline")) underline = true;
				else break;
			p.PassChar(';');

			for(int i = 0; i < PlantUML::PUML_HL_COUNT; i++) {
				if(id == puml_style_name[i]) {
					currInk   = &PlantUmlSyntax::puml_style_ink[i];
					currPaper = &PlantUmlSyntax::puml_style_paper[i];
					break;
				}
			}
			
			if (currInk) {
				currInk->color     = inkColor;
				currInk->bold      = bold;
				currInk->italic    = italic;
				currInk->underline = underline;
			}
			if (currPaper) {
				currPaper->color = paperColor;
			}
		}
	}
	catch(CParser::Error) {
		loadDefaultSettings();
	}
}

String PlantUmlSyntax::StoreColorSettings()
{
	String r;
	// Add title line
	r << "//  STYLE_NAME                 ink_color                 paper_color                 bold   italic   underline;\r\n";
	r << "//  ----------                 ---------                 -----------                 ----   ------   ---------;\r\n";

	// Add styles config
	for(int i = 0; i < PlantUML::PUML_HL_COUNT; i++) {
		const HlStyle& ink = puml_style_ink[i];
		const HlStyle& paper = puml_style_paper[i];
		r << Upp::Format("%-30s", puml_style_name[i]) << ' ' << Upp::Format("%-25s", Upp::FormatColor(ink.color)) << ' ' << Upp::Format("%-25s", Upp::FormatColor(paper.color));
		if(ink.bold)      r << "   bold";
		else              r << "       ";

		if(ink.italic)	  r << "   italic";
		else              r << "         ";

		if(ink.underline) r << "   underline";
		else              r << "            ";

		r << ";\r\n";
	}
	return r;
}
