namespace GraphDraw_ns
{
	// ============================
	// CustomData   CLASS
	// ============================
	class CustomData
	{
		public:
			CustomData() {}
			virtual ~CustomData() {}
			
			virtual bool Intersects(const RectGraph& rect) const = 0;
			virtual bool Contains(const PointGraph& pt) const = 0;
			virtual void PaintDataPoint(BufferPainter& dw, int scale, const CoordinateConverter* xCoordConv, const CoordinateConverter* yCoordConv ) = 0;
			virtual void PaintDataPoint(BufferPainter& dw, int scale, int style, const CoordinateConverter* xCoordConv, const CoordinateConverter* yCoordConv ) {
				// style: CTRL_NORMAL, CTRL_HOT, CTRL_PRESSED, CTRL_DISABLED, CTRL_CHECKED, CTRL_HOTCHECKED
				PaintDataPoint(dw, scale, xCoordConv, yCoordConv);
			}
	};


	// ============================
	// CustomDataSource   CLASS
	// ============================
	class CustomDataSource
	{
		public:
			CustomDataSource() {}
			virtual ~CustomDataSource() {}
			virtual unsigned int GetCount() const = 0;
			virtual const CustomData& Get(unsigned int dataIndex) const = 0;
			virtual       CustomData& Get(unsigned int dataIndex) = 0;

			virtual void PaintSerie(BufferPainter& dw, int scale, const CoordinateConverter* xCoordConv, const CoordinateConverter* yCoordConv )
			{
				unsigned int count=GetCount();
				const RectGraph gr(xCoordConv->getGraphMin(), yCoordConv->getGraphMin(), xCoordConv->getGraphMax(), yCoordConv->getGraphMax());
				for (unsigned int c=0; c<count; ++c) {
					// paint only visible data points
					if (Get(c).Intersects(gr)) Get(c).PaintDataPoint(dw, scale, xCoordConv, yCoordConv);
				}
			}

			virtual void PaintOne(unsigned int index, int style, BufferPainter& dw, int scale, const CoordinateConverter* xCoordConv, const CoordinateConverter* yCoordConv )
			{
				// style: CTRL_NORMAL, CTRL_HOT, CTRL_PRESSED, CTRL_DISABLED, CTRL_CHECKED, CTRL_HOTCHECKED
				if (index < GetCount()) {
					Get(index).PaintDataPoint(dw, scale, xCoordConv, yCoordConv);
				}
			}
			
			virtual bool Contains(PointGraph p, unsigned int& pIndexOut) const {
				unsigned int count=GetCount();
				for (unsigned int c=0; c<count; ++c) {
					if (Get(c).Contains(p)) {
						pIndexOut = c;
						return true;
					}
				}
				return false;
			}

			// ===========================
			// stubs for Ctrl enabling
			// ===========================
			virtual bool IsCtrlable() { return false; }
			
			virtual bool LeftDown    (unsigned int pIndex, PointGraph p, dword keyflags) { return false; }
			virtual bool LeftUp      (unsigned int pIndex, PointGraph p, dword keyflags) { return false; }
			virtual bool LeftDouble  (unsigned int pIndex, PointGraph p, dword keyflags) { return false; }
			virtual bool LeftTriple  (unsigned int pIndex, PointGraph p, dword keyflags) { return false; }
			virtual bool LeftRepeat  (unsigned int pIndex, PointGraph p, dword keyflags) { return false; }
			virtual bool LeftDrag    (unsigned int pIndex, PointGraph p, dword keyflags) { return false; }
			virtual bool LeftHold    (unsigned int pIndex, PointGraph p, dword keyflags) { return false; }

			virtual bool RightDown   (unsigned int pIndex, PointGraph p, dword keyflags) { return false; }
			virtual bool RightUp     (unsigned int pIndex, PointGraph p, dword keyflags) { return false; }
			virtual bool RightDouble (unsigned int pIndex, PointGraph p, dword keyflags) { return false; }
			virtual bool RightTriple (unsigned int pIndex, PointGraph p, dword keyflags) { return false; }
			virtual bool RightRepeat (unsigned int pIndex, PointGraph p, dword keyflags) { return false; }
			virtual bool RightDrag   (unsigned int pIndex, PointGraph p, dword keyflags) { return false; }
			virtual bool RightHold   (unsigned int pIndex, PointGraph p, dword keyflags) { return false; }

			virtual bool MiddleDown  (unsigned int pIndex, PointGraph p, dword keyflags) { return false; }
			virtual bool MiddleUp    (unsigned int pIndex, PointGraph p, dword keyflags) { return false; }
			virtual bool MiddleDouble(unsigned int pIndex, PointGraph p, dword keyflags) { return false; }
			virtual bool MiddleTriple(unsigned int pIndex, PointGraph p, dword keyflags) { return false; }
			virtual bool MiddleRepeat(unsigned int pIndex, PointGraph p, dword keyflags) { return false; }
			virtual bool MiddleDrag  (unsigned int pIndex, PointGraph p, dword keyflags) { return false; }
			virtual bool MiddleHold  (unsigned int pIndex, PointGraph p, dword keyflags) { return false; }

			virtual bool MouseMove  (unsigned int pIndex, PointGraph p, dword keyflags) { return false; }
			virtual bool MouseWheel (unsigned int pIndex, PointGraph p, int zdelta, dword keyflags) { return false; }
			
			virtual bool CursorImage(unsigned int pIndex, PointGraph p, dword keyflags, Image& outImg ) { return false; }
		
			virtual bool ContextMenu(unsigned int pIndex, Bar& bar) { return false; }
			//virtual void FitToData(FitToDataStrategy fitStrategy) {}

	};

	template <class CUSTOM_DATA_TYPE>
	class CustomArrayDataSource : public CustomDataSource {
		protected:
			Array<CUSTOM_DATA_TYPE> dataArray;
			
		public:
			CustomArrayDataSource() {}
			virtual ~CustomArrayDataSource() {}
			
			virtual unsigned int GetCount() const { return dataArray.GetCount(); }
			virtual const CustomData& Get(unsigned int dataIndex) const { return dataArray[dataIndex]; }
			virtual       CustomData& Get(unsigned int dataIndex) { return dataArray[dataIndex]; }

			
			CUSTOM_DATA_TYPE& operator[](int dataIndex) { return dataArray[dataIndex]; }
	};

	
	// ============================
	// CustomSeriesGroup<>   CLASS
	// ============================
	template<class CUSTOM_DATA_SOURCE>
	struct TCreateCDSPropertiesDlg {
		void* operator () (CUSTOM_DATA_SOURCE) const { return 0; }
	};

	class CustomDataSourceDecorator : public Moveable<CustomDataSourceDecorator> {
		private:
			CustomDataSource* cds;
			const CoordinateConverter* xCoordConv;
			const CoordinateConverter* yCoordConv;
			Function<void* ()> createPropertiesDlgCB;


		public:
			CustomDataSourceDecorator() : cds(0), xCoordConv(0), yCoordConv(0) {}
			CustomDataSourceDecorator(const CustomDataSourceDecorator& ds) : cds(ds.cds), xCoordConv(ds.xCoordConv), yCoordConv(ds.yCoordConv) {}
			~CustomDataSourceDecorator() {}

			
			template <class CUST_DATA_SOURCE, typename CREATE_FUNCTOR = TCreateCDSPropertiesDlg<CUST_DATA_SOURCE> >
			void SetDataSource(CUST_DATA_SOURCE* ds)
			{
				cds = ds;
				//createPropertiesDlgCB << [&] () -> void* { return CREATE_FUNCTOR()(ds); }; // TODO
			}

			void SetXCoordConverter(CoordinateConverter& c) { xCoordConv = &c; }
			void SetYCoordConverter(CoordinateConverter& c) { yCoordConv = &c; }

			inline void PaintSerie(BufferPainter& dw, int scale) { cds->PaintSerie(dw, scale, xCoordConv, yCoordConv); }
			
			inline PointGraph ToGraph(const PointScreen& p) {
				PointGraph res;
				res.x = xCoordConv->toGraph(p.x);
				res.y = yCoordConv->toGraph(p.y);
				return res;
			}
			
			inline bool Contains(PointScreen p, unsigned int& pIndexOut)  { return cds->Contains( ToGraph(p), pIndexOut); } // return index of point which is at 'p' coord

			inline bool IsCtrlable()                                                      { return cds->IsCtrlable(); }
//			virtual void MouseEnter(int pIndex, PointScreen p, dword keyflags)  {}
//			virtual void MouseLeave()                                           {}
			inline bool LeftDown   (unsigned int pIndex, PointScreen p, dword keyflags)    { return cds->LeftDown    ( pIndex, ToGraph(p), keyflags); }
			inline bool LeftUp     (unsigned int pIndex, PointScreen p, dword keyflags)    { return cds->LeftUp      ( pIndex, ToGraph(p), keyflags); }
			inline bool LeftDouble (unsigned int pIndex, PointScreen p, dword keyflags)    { return cds->LeftDouble  ( pIndex, ToGraph(p), keyflags); }
			inline bool LeftTriple (unsigned int pIndex, PointScreen p, dword keyflags)    { return cds->LeftTriple  ( pIndex, ToGraph(p), keyflags); }
			inline bool LeftRepeat (unsigned int pIndex, PointScreen p, dword keyflags)    { return cds->LeftRepeat  ( pIndex, ToGraph(p), keyflags); }
			inline bool LeftDrag   (unsigned int pIndex, PointScreen p, dword keyflags)    { return cds->LeftDrag    ( pIndex, ToGraph(p), keyflags); }
			inline bool LeftHold   (unsigned int pIndex, PointScreen p, dword keyflags)    { return cds->LeftHold    ( pIndex, ToGraph(p), keyflags); }

			inline bool RightDown   (unsigned int pIndex, PointScreen p, dword keyflags)    { return cds->LeftDown    ( pIndex, ToGraph(p), keyflags); }
			inline bool RightUp     (unsigned int pIndex, PointScreen p, dword keyflags)    { return cds->LeftUp      ( pIndex, ToGraph(p), keyflags); }
			inline bool RightDouble (unsigned int pIndex, PointScreen p, dword keyflags)    { return cds->LeftDouble  ( pIndex, ToGraph(p), keyflags); }
			inline bool RightTriple (unsigned int pIndex, PointScreen p, dword keyflags)   { return cds->LeftTriple  ( pIndex, ToGraph(p), keyflags); }
			inline bool RightRepeat (unsigned int pIndex, PointScreen p, dword keyflags)   { return cds->LeftRepeat  ( pIndex, ToGraph(p), keyflags); }
			inline bool RightDrag   (unsigned int pIndex, PointScreen p, dword keyflags)   { return cds->LeftDrag    ( pIndex, ToGraph(p), keyflags); }
			inline bool RightHold   (unsigned int pIndex, PointScreen p, dword keyflags)   { return cds->LeftHold    ( pIndex, ToGraph(p), keyflags); }

			inline bool MiddleDown  (unsigned int pIndex, PointScreen p, dword keyflags)   { return cds->MiddleDown  ( pIndex, ToGraph(p), keyflags); }
			inline bool MiddleUp    (unsigned int pIndex, PointScreen p, dword keyflags)   { return cds->MiddleUp    ( pIndex, ToGraph(p), keyflags); }
			inline bool MiddleDouble(unsigned int pIndex, PointScreen p, dword keyflags)   { return cds->MiddleDouble( pIndex, ToGraph(p), keyflags); }
			inline bool MiddleTriple(unsigned int pIndex, PointScreen p, dword keyflags)   { return cds->MiddleTriple( pIndex, ToGraph(p), keyflags); }
			inline bool MiddleRepeat(unsigned int pIndex, PointScreen p, dword keyflags)   { return cds->MiddleRepeat( pIndex, ToGraph(p), keyflags); }
			inline bool MiddleDrag  (unsigned int pIndex, PointScreen p, dword keyflags)   { return cds->MiddleDrag  ( pIndex, ToGraph(p), keyflags); }
			inline bool MiddleHold  (unsigned int pIndex, PointScreen p, dword keyflags)   { return cds->MiddleHold  ( pIndex, ToGraph(p), keyflags); }

			inline bool MouseMove  (unsigned int pIndex, PointScreen p, dword keyflags)             { return cds->MouseMove(  pIndex, ToGraph(p), keyflags); }
			inline bool MouseWheel (unsigned int pIndex, PointScreen p, int zdelta, dword keyflags) { return cds->MouseWheel( pIndex, ToGraph(p), zdelta, keyflags); }

			inline bool CursorImage(unsigned int pIndex, PointGraph p, dword keyflags, Image& outImg ) { return cds->CursorImage(pIndex, ToGraph(p), keyflags, outImg); }

			inline bool ContextMenu(unsigned int pIndex, Bar& bar) { return cds->ContextMenu(pIndex, bar); }
			

	};
	
	typedef Vector<CustomDataSourceDecorator> CustomDataSourceVector;
	

	template<class DERIVED, class BASE>
	class CustomSeriesGroup : public BASE
	{
		private:
			bool _isCustomSeriesModified;
		public:
			typedef CustomSeriesGroup CLASSNAME;
			typedef BASE _B;
			CustomDataSourceVector customDataSourceProxyVector;
			
			CustomSeriesGroup() : _isCustomSeriesModified(true) {}
			virtual ~CustomSeriesGroup() {}


			inline void SetModifyCustomSeries()   { _isCustomSeriesModified = true;  }
			inline void ClearModifyCustomSeries() { _isCustomSeriesModified = false; }
			inline bool IsModifyCustomSeries() const { return _isCustomSeriesModified;  }
			
			template <class CUST_DATA_SOURCE>
			void AddCustomSeries(CUST_DATA_SOURCE& sc) {
				AddCustomSeries(&sc);
			}

			void ClearAllCustomSeries() {
				customDataSourceProxyVector.Clear();
			}

			template <class CUST_DATA_SOURCE>
			void AddCustomSeries(CUST_DATA_SOURCE* ds) {
				CustomDataSourceDecorator& ccds = customDataSourceProxyVector.Add();
				ccds.SetDataSource(ds);
				ccds.SetXCoordConverter((*BASE::_currentXConverter)); // TODO changer gestion _currentXConverter  ==> refactorer SeriesGroup
				ccds.SetYCoordConverter(*(BASE::_currentYConverter));
				SetModifyCustomSeries();
			}
			
			void PaintAllCustomSeries(BufferPainter& dw, int scale) {
				CustomDataSourceVector::iterator iter = customDataSourceProxyVector.Begin();
				CustomDataSourceVector::const_iterator endIter = customDataSourceProxyVector.End();
				for (; iter != endIter; ++iter) {
					(*iter).PaintSerie(dw, scale);
				}
			}
	};
};

