/*
 * GridAxisDraw.h
 *
 *  Created on: 6 avr. 2012
 *      Author: didier
 */

#ifndef GRIDAXISDRAW_H_
#define GRIDAXISDRAW_H_


#include "TickMark.h"

namespace GraphDraw_ns
{

	// ============================
	//    GridAxisDraw   CLASS
	// ============================
	typedef enum {
		AXIS_TEXT_FORMAT_STD,
		AXIS_TEXT_FORMAT_LOG,
		AXIS_TEXT_FORMAT_DATE,
		AXIS_TEXT_FORMAT_TIME,
	} AxisTextFormat;

	struct GridAxisDraw_GEStyle : ChStyle<GridAxisDraw_GEStyle>, GraphElement::GEStyle {
			int       axisWidth;
			Color     axisColor;
			Font      axisTextFont;
			Color     axisTextColor;
			Color     axisTickColor;
			Color     gridColor;
			Function<TickMark* ()> primaryTickFactory;
			Function<TickMark* ()> secondaryTickFactory;
	};

	class GridAxisDraw : public GraphElement
	{
	public:
		typedef GraphElement  _B;
		typedef GridAxisDraw  CLASSNAME;

		typedef GridAxisDraw_GEStyle Style;
		
		static const Style& StyleDefault();
		static const Style& StyleDefault2(); // default for X2,X3,...,Y2,Y3,....
		GridAxisDraw&  SetStyle(const Style& s) {
			style = &s;
			_tickMarks[0] = style->primaryTickFactory();
			_tickMarks[1] = style->secondaryTickFactory();
			return *this;
		}

	protected:
		const Style *style;
		
	public:
		
//		protected:
		CoordinateConverter& _coordConverter;
		GridStepManager _gridStepManager;

		enum {NB_TICKMARKS = 2 };
		One<TickMark> _tickMarks[NB_TICKMARKS]; // One<>  used for ownership (manages delete)
		TypeFormatTextCbk _formatTextCbk;
		Sizef _meanTickTextsz;
		Sizef _tmpTextSize;
		unsigned int _nbMeanValues;


	public:
		GridAxisDraw(CoordinateConverter& coordConv)
		: _coordConverter( coordConv )
		, _gridStepManager( coordConv )
		, _formatTextCbk(THISBACK(FormatAsDouble))
		, _meanTickTextsz(10,10)
		, _tmpTextSize(10,10)
		, _nbMeanValues(0)
		{
			SetStyle(StyleDefault());
			_B::DisablePos(FLOAT_OVER_GRAPH);
			PrePaint(1);
		}
	private:
		GridAxisDraw(GridAxisDraw& p)
		: _B(p)
		, style(p.style)
		, _coordConverter( p._coordConverter )
		, _gridStepManager( p._coordConverter )
		, _formatTextCbk( p._formatTextCbk )
		, _meanTickTextsz( p._meanTickTextsz )
		, _tmpTextSize(10,10)
		, _nbMeanValues(0)
		{
			SetStyle(*p.style);
			PrePaint(1);
		}
	public:
		virtual ~GridAxisDraw() {}
		
		virtual void PrePaint( int scale ) {
			//RLOGBLOCK( "GridAxisDraw::PrePaint()" );
			UpdateTickTextData(scale);
			if ( _B::IsHorizontal() ) { _gridStepManager.SetTextMaxSize( _meanTickTextsz.cx ); }
			else                      { _gridStepManager.SetTextMaxSize( _meanTickTextsz.cy ); }
			_gridStepManager.UpdateGridSteps();
		}
		

		virtual void  SetElementPos(ElementPosition v) {
			if (v==LEFT_OF_GRAPH || v==RIGHT_OF_GRAPH) {
				_B::DisablePos(BOTTOM_OF_GRAPH);
				_B::DisablePos(TOP_OF_GRAPH);
			} else {
				_B::DisablePos(LEFT_OF_GRAPH);
				_B::DisablePos(RIGHT_OF_GRAPH);
			}
			_B::SetElementPos(v);
		}

		
		virtual void FitToData(FitToDataStrategy fitStrategy) {
			TypeVectorSeries& v_series = _B::_parent->GetSeries();
			double lmin = -DOUBLE_NULL;
			double lmax =  DOUBLE_NULL;
			bool doFitToData = false;
			switch(_B::GetElementPos()) {
				case LEFT_OF_GRAPH:
				case RIGHT_OF_GRAPH:
					for (int c=0; c<v_series.GetCount(); ++c) {
						if ( ( (fitStrategy==ALL_SERIES)  || (v_series[c].show)) && (v_series[c].yConverter == &_coordConverter) && !v_series[c].PointsData()->IsExplicit() && !v_series[c].PointsData()->IsParam() ) {
							lmin = min (lmin, v_series[c].PointsData()->MinY());
							lmax = max (lmax, v_series[c].PointsData()->MaxY());
							doFitToData = true;
						}
					}
					break;
				case BOTTOM_OF_GRAPH:
				case TOP_OF_GRAPH:
					for (int c=0; c<v_series.GetCount(); ++c) {
						if ( ( (fitStrategy==ALL_SERIES)  || (v_series[c].show)) && (v_series[c].xConverter == &_coordConverter) && !v_series[c].PointsData()->IsExplicit() && !v_series[c].PointsData()->IsParam() ) {
							lmin = min (lmin, v_series[c].PointsData()->MinX());
							lmax = max (lmax, v_series[c].PointsData()->MaxX());
							doFitToData = true;
						}
					}
					break;
				case FLOAT_OVER_GRAPH:
					break;
			}
			if (doFitToData) {
				_coordConverter.updateGraphSize( lmin, lmax );
			}
		}

		GridStepManager& GetGridStepManager()                       { return _gridStepManager; }

		inline CLASSNAME& setMajorTickMark(TickMark* v)             { _tickMarks[0] = v; return *this;  }
		inline CLASSNAME& setMinorTickMark(TickMark* v)             { _tickMarks[1] = v; return *this;  }
		
		inline CLASSNAME& setAxisTextFormat(TypeFormatTextCbk v)    { _formatTextCbk = v; return *this;  }
		
		CLASSNAME& setAxisTextFormat(AxisTextFormat v) {
			switch(v) {
				case AXIS_TEXT_FORMAT_STD:  resetAxisTextFormat() ; break;
				case AXIS_TEXT_FORMAT_LOG:  setAxisLogFormat(); break;
				case AXIS_TEXT_FORMAT_DATE: setAxisDateFormat(); break;
				case AXIS_TEXT_FORMAT_TIME: setAxisTimeFormat(); break;
			}
			return *this;
		}
		
		inline CLASSNAME& resetAxisTextFormat() {
			_formatTextCbk = THISBACK(FormatAsDouble);
			_gridStepManager.setStdGridSteps();
			return *this;
		}
		
		inline CLASSNAME& setAxisLogFormat(TypeFormatTextCbk cbk=TypeFormatTextCbk()) {
			if ( cbk ) _formatTextCbk = cbk;
			else       _formatTextCbk = THISBACK(FormatAsLog10);
			_gridStepManager.setLogGridSteps();
			return *this;
		}

		inline CLASSNAME& setAxisDateFormat( TypeFormatTextCbk cbk=TypeFormatTextCbk() ) {
			if ( cbk ) _formatTextCbk = cbk;
			else       _formatTextCbk = THISBACK(FormatAsDate);
			_gridStepManager.setDateGridSteps();
			return *this;
		}
		
		inline CLASSNAME& setAxisTimeFormat( TypeFormatTextCbk cbk=TypeFormatTextCbk() ) {
			if ( cbk ) _formatTextCbk = cbk;
			else       _formatTextCbk = THISBACK(FormatAsTime);
			_gridStepManager.setTimeGridSteps();
			return *this;
		}


		inline int GetMajorTickLength()       { return _tickMarks[0]->GetTickLength(); }
		inline void SetMajorTickLength(int v) { _tickMarks[0]->SetTickLength(v); }

		inline int GetMinorTickLength()       { return _tickMarks[1]->GetTickLength(); }
		inline void SetMinorTickLength(int v) { _tickMarks[1]->SetTickLength(v); }

		inline CoordinateConverter& GetCoordConverter() { return _coordConverter;  }

		inline       GridStepIterator GridStepBegin(void) { return  _gridStepManager.Begin(); }
		inline const_GridStepIterator GridStepEnd(void)   { return  _gridStepManager.End(); }

		void FormatAsDouble( const const_GridStepIterator& iter, String& output ) {
			if (iter->drawTickText) {
				output = FormatDouble( iter->stepGraphValue );
			}
		}

		void FormatAsLog10( const const_GridStepIterator& iter, String& output ) {
			if (iter->drawTickText) {
				const double d = iter->stepGraphValue;
				if (d < 0.001 || d > 9999) output << FormatDoubleExp(d, 3);//, FD_SIGN_EXP | FD_SIGN | FD_ZERO);
				else                       output << d ;
			}
		}

		void FormatAsDate( const const_GridStepIterator& value, String& output ) {
			Date dat;
			dat.Set(int(value->stepGraphValue));
			output = Format("%d/%d/%d",dat.day, dat.month, dat.year);
		}

		void FormatAsTime( const const_GridStepIterator& value, String& output ) {
			Time time;
			time.Set(Upp::int64( value->stepGraphValue ));
			
			TypeGraphCoord range = value.getGraphRange();
			Time timeStep;
			timeStep.Set( Upp::int64(range / value.getNbSteps()) );
			//LOG("TimeStep = " << timeStep);
			
			if (range > 7*365*24*60*60) { // 7 years
				output << "\1[1= " << time.year << "]";
			}
			else if (range > 182.5*24*60*60) { // 6 months
				if (((time.month == 1) && (time.day == 1) )
					|| value.isFirst() || value.isLast() )
				   {
					switch(_B::GetElementPos()){
						case LEFT_OF_GRAPH:
							output << "\1[1* " << time.year;
							output << " ][1 " << MonName(time.month-1) << "]";
							break;
						case RIGHT_OF_GRAPH:
							output << "\1[1 " << MonName(time.month-1) ;
							output << " ][1* " << time.year << "]";
							break;
						case BOTTOM_OF_GRAPH:
							output << "\1[1= " << MonName(time.month-1) ;
							output << "&][1*= " << time.year << "]";
							break;
						case FLOAT_OVER_GRAPH:
						case TOP_OF_GRAPH:
							output << "\1[1*= " << time.year;
							output << "&][1= " << MonName(time.month-1) << "]";
							break;
					}
				}
				else {
					output << "\1[1 " << MonName(time.month-1) << "]";
				}
			}
			else if (range > 7*24*60*60) { // 7 days
				if ( ((time.day == 1) && (time.hour==0))
						|| value.isFirst() || value.isLast() )
				{
					switch(_B::GetElementPos()){
						case LEFT_OF_GRAPH:
							output << "\1[1* " << time.year;
							output << " ][1 " << int(time.day) << " " << MonName(time.month-1) << "]";
							break;
						case RIGHT_OF_GRAPH:
							output << "\1[1 " << int(time.day) << " " << MonName(time.month-1) ;
							output << " ][1* " << time.year << "]";
							break;
						case BOTTOM_OF_GRAPH:
							// [ [= 25&][= [* Mar 2014]]]
							output << "\1[1= " << int(time.day) ;
							output << "&][1*= " << MonName(time.month-1) << " " << time.year << "]";
							break;
						case FLOAT_OVER_GRAPH:
						case TOP_OF_GRAPH:
							output << "\1[1*= " << time.year;
							output << "&][1= " << int(time.day) << " " << MonName(time.month-1) << "]";
							break;
					}
				}
				else {
					output << "\1[1 " << int(time.day) << "]";
				}
			}
			else {
				if ( value.isFirst() || value.isLast() ) {
					switch(_B::GetElementPos()) {
						case LEFT_OF_GRAPH:
							output << "\1[1* " << time.year;
							output << " ][1 " << int(time.day) << " " << MonName(time.month-1) << "]";
							output << "[1 " << FormatTime( time, " h`hmm:ss" ) << "]";
							break;
						case RIGHT_OF_GRAPH:
							output << "\1[1 " << FormatTime( time, "h`hmm:ss " ) << "]";
							output << "[1 " << int(time.day) << " " << MonName(time.month-1) ;
							output << " ][1* " << time.year << "]";
							break;
						case BOTTOM_OF_GRAPH:
							output << "\1[1= " << FormatTime( time, "h`hmm:ss" ) << "&]";
							output << "[1= " << int(time.day) << " " << MonName(time.month-1) ;
							output << "&][1*= " << time.year << "]";
							break;
						case FLOAT_OVER_GRAPH:
						case TOP_OF_GRAPH:
							output << "\1[1*= " << time.year;
							output << "&][1= " << int(time.day) << " " << MonName(time.month-1) << "&]";
							output << "[1= " << FormatTime( time, "h`hmm:ss" ) << "]";
							break;
					}
				}
				else {
					if (range > 24*60*60) { // 24 hours
/*						if ( (range % range) == 0 ) { // multiple of days
						switch(_B::GetElementPos()) {
							case LEFT_OF_GRAPH:
								output << "\1[1 " << int(time.day) << " " << MonName(time.month-1) << "]";
								break;
							case RIGHT_OF_GRAPH:
								output << "\1[1 " << int(time.day) << " " << MonName(time.month-1) ;
								output << " ]";
								break;
							case BOTTOM_OF_GRAPH:
								output << "\1[1= " << int(time.day) << " " << MonName(time.month-1) ;
								output << "&]";
								break;
							case FLOAT_OVER_GRAPH:
							case TOP_OF_GRAPH:
								output << "\1[1= " << int(time.day) << " " << MonName(time.month-1) << "&]";
								break;
							}
						}
						else {
*/							switch(_B::GetElementPos()){
								case LEFT_OF_GRAPH:
									output << "\1[1 " << int(time.day) << " " << MonName(time.month-1) << "]";
									if (time.hour != 0 || time.minute!=0 || time.second!=0) {
										output << "[1 " << FormatTime( time, " h`hmm:ss" ) << "]";
									}
									break;
								case RIGHT_OF_GRAPH:
									output << "\1";
									if (time.hour != 0 || time.minute!=0 || time.second!=0) {
										output <<"[1 " << FormatTime( time, "h`hmm:ss " ) << "]";
									}
									output << "[1 " << int(time.day) << " " << MonName(time.month-1) ;
									output << " ]";
									break;
								case BOTTOM_OF_GRAPH:
									output << "\1";
									if (time.hour != 0 || time.minute!=0 || time.second!=0) {
										output << "[1= " << FormatTime( time, "h`hmm:ss" ) << "&]";
									}
									output << "[1= " << int(time.day) << " " << MonName(time.month-1) ;
									output << "&]";
									break;
								case FLOAT_OVER_GRAPH:
								case TOP_OF_GRAPH:
									output << "\1[1= " << int(time.day) << " " << MonName(time.month-1) << "&]";
									if (time.hour != 0 || time.minute!=0 || time.second!=0) {
										output << "[1= " << FormatTime( time, "h`hmm:ss" ) << "]";
									}
									break;
							}
//						}
					}
					else {
						output << "\1[1 " << FormatTime( time, "h`hmm:ss" ) << "]";
					}
				}
			}
		}


		void UpdateTickTextData( int scale ) {
			Sizef tmpSize(0,0);
			int nbSizeValues = 0;
			 _gridStepManager.UpdateGridSteps();

			GridStepIterator iter = GridStepBegin();
			const_GridStepIterator endIter = GridStepEnd();

			Font scaledAxisTextFont( style->axisTextFont );
			if (scale>1) scaledAxisTextFont.Height(scale*scaledAxisTextFont.GetHeight());
			while ( iter != endIter )
			{
				iter->text.Clear();
				_formatTextCbk( iter, iter->text );
				
				if (!iter->text.IsEmpty()) {
					iter->textSize = GraphDraw_ns::GetSmartTextSize(iter->text , scaledAxisTextFont, scale);
					tmpSize.cx += iter->textSize.cx;
					tmpSize.cy += iter->textSize.cy;
					++nbSizeValues;
				}
				++iter;
			}
			tmpSize /= nbSizeValues;
			_meanTickTextsz = tmpSize;
		}

		template<int GRAPH_SIDE>
		void PaintTickText(Draw& dw,  const const_GridStepIterator& value, TypeScreenCoord x, TypeScreenCoord y, const Color color, const Font scaledFont, int scale) {
			Upp::String text;
			_formatTextCbk( value, text );
			if (!text.IsEmpty()) {
				Size sz = GraphDraw_ns::GetSmartTextSize(text, scaledFont, scale);
				_tmpTextSize += sz;
				++_nbMeanValues;

				if (GRAPH_SIDE == LEFT_OF_GRAPH) {
					GraphDraw_ns::DrawSmartText(dw, x-sz.cx, y-(sz.cy/2), sz.cx, text, scaledFont, color, scale);
				}
				else if (GRAPH_SIDE == BOTTOM_OF_GRAPH) {
					GraphDraw_ns::DrawSmartText(dw, x-(sz.cx/2), y, sz.cx, text, scaledFont, color, scale);
				}
				else if (GRAPH_SIDE == RIGHT_OF_GRAPH) {
					GraphDraw_ns::DrawSmartText( dw, x, y-(sz.cy/2), sz.cx, text, scaledFont, color, scale);
				}
				else {
					GraphDraw_ns::DrawSmartText( dw,  x-(sz.cx/2), y-sz.cy ,sz.cx,  text, scaledFont, color, scale);
				}
			}
		}

		virtual void PaintAxisLeft(Draw& dw, const int scale, const_GridStepIterator iter, const_GridStepIterator endIter)
		{
			Font scaledAxisTextFont( style->axisTextFont );
			if (scale>1) scaledAxisTextFont.Height(scale*scaledAxisTextFont.GetHeight());
			dw.DrawLineOp(_B::GetElementWidth()*scale, _coordConverter.getScreenMin(), _B::GetElementWidth()*scale, _coordConverter.getScreenMax(), style->axisWidth*scale, style->axisColor );
			while ( iter != endIter)
			{
				if (_tickMarks[iter->tickLevel].IsEmpty()) {
					PaintTickText<LEFT_OF_GRAPH>(dw, iter, (_B::GetElementWidth()-8)*scale, iter, style->axisTextColor, scaledAxisTextFont, scale);
				} else {
					_tickMarks[iter->tickLevel]->Paint(dw, LEFT_OF_GRAPH, scale, _B::GetElementWidth()*scale, iter, style->axisTickColor );
					PaintTickText<LEFT_OF_GRAPH>(dw, iter, (_B::GetElementWidth()-_tickMarks[0]->GetTickLength()-2)*scale, iter, style->axisTextColor, scaledAxisTextFont, scale);
				}
				++iter;
			}
		}

		virtual void PaintAxisRight(Draw& dw, const int scale, const_GridStepIterator iter, const_GridStepIterator endIter)
		{
			Font scaledAxisTextFont( style->axisTextFont );
			if (scale>1) scaledAxisTextFont.Height(scale*scaledAxisTextFont.GetHeight());
			dw.DrawLineOp(0, _coordConverter.getScreenMin(), 0, _coordConverter.getScreenMax(), style->axisWidth*scale, style->axisColor);
			while ( iter != endIter)
			{
				if (_tickMarks[iter->tickLevel].IsEmpty())
				{
					PaintTickText<RIGHT_OF_GRAPH>(dw, iter, 8, iter, style->axisTextColor, scaledAxisTextFont, scale);
				} else {
					_tickMarks[iter->tickLevel]->Paint(dw, RIGHT_OF_GRAPH, scale, 0, iter, style->axisTickColor );
					PaintTickText<RIGHT_OF_GRAPH>(dw, iter, (_tickMarks[0]->GetTickLength()+2)*scale, iter, style->axisTextColor, scaledAxisTextFont, scale);
				}
				++iter;
			}
		}

		virtual void PaintAxisBottom(Draw& dw, const int scale, const_GridStepIterator iter, const_GridStepIterator endIter)
		{
			Font scaledAxisTextFont( style->axisTextFont );
			if (scale>1) scaledAxisTextFont.Height(scale*scaledAxisTextFont.GetHeight());
			dw.DrawLineOp(_coordConverter.getScreenMin(), 0, _coordConverter.getScreenMax(),0 , style->axisWidth*scale, style->axisColor );
			while ( iter != endIter )
			{
				if (_tickMarks[iter->tickLevel].IsEmpty()) {
					PaintTickText<BOTTOM_OF_GRAPH>(dw, iter, iter, 4, style->axisTextColor, scaledAxisTextFont, scale);
				} else {
					_tickMarks[iter->tickLevel]->Paint(dw, BOTTOM_OF_GRAPH, scale, iter, 0, style->axisTickColor );
					PaintTickText<BOTTOM_OF_GRAPH>(dw, iter, iter, (_tickMarks[0]->GetTickLength()+2)*scale, style->axisTextColor, scaledAxisTextFont, scale);
				}
				++iter;
			}
		}

		virtual void PaintAxisTop(Draw& dw, const int scale, const_GridStepIterator iter, const_GridStepIterator endIter)
		{
			Font scaledAxisTextFont( style->axisTextFont );
			if (scale>1) scaledAxisTextFont.Height(scale*scaledAxisTextFont.GetHeight());
			dw.DrawLineOp(_coordConverter.getScreenMin(), _B::GetElementWidth()*scale, _coordConverter.getScreenMax(), _B::GetElementWidth()*scale, style->axisWidth*scale, style->axisColor );
			while ( iter != endIter)
			{
				if (_tickMarks[iter->tickLevel].IsEmpty()) {
					PaintTickText<TOP_OF_GRAPH>(dw, iter, iter, (_B::GetElementWidth()-4)*scale, style->axisTextColor, scaledAxisTextFont, scale);
				} else {
					_tickMarks[iter->tickLevel]->Paint(dw, TOP_OF_GRAPH, scale, iter, _B::GetElementWidth()*scale, style->axisTickColor );
					PaintTickText<TOP_OF_GRAPH>(dw, iter, iter, (_B::GetElementWidth()-_tickMarks[0]->GetTickLength()-2)*scale, style->axisTextColor, scaledAxisTextFont, scale);
				}
				++iter;
			}
		}

		virtual void PaintVGrid(Draw& dw, int xRange, const_GridStepIterator iter, const_GridStepIterator endIter)
		{
			if ( !style->gridColor.IsNullInstance())
			{
				while( iter != endIter)
				{
					dw.DrawLineOp( 0, iter, xRange, iter, 1, style->gridColor );
					++iter;
				}
			}
		}

		virtual void PaintHGrid(Draw& dw, int yRange, const_GridStepIterator iter, const_GridStepIterator endIter)
		{
			if ( !style->gridColor.IsNullInstance())
			{
				const_GridStepIterator iter = GridStepBegin();
				const_GridStepIterator endIter = GridStepEnd();
				while( iter != endIter)
				{
					dw.DrawLineOp( iter, 0, iter, yRange, 1, style->gridColor );
					++iter;
				}
			}
		}

		virtual void PaintOnPlot_underData(Draw& dw, int otherWidth, int scale)
		{
			const_GridStepIterator iter = GridStepBegin();
			const_GridStepIterator endIter = GridStepEnd();
			switch(_B::GetElementPos()){
				case LEFT_OF_GRAPH:
				case RIGHT_OF_GRAPH:
					PaintVGrid(dw, otherWidth, iter, endIter);
					break;
				case BOTTOM_OF_GRAPH:
				case TOP_OF_GRAPH:
					PaintHGrid(dw, otherWidth, iter, endIter);
					break;
				case FLOAT_OVER_GRAPH:
					break;
			}
		}

		virtual void PaintElement(Draw& dw, int scale)
		{
			_tmpTextSize.Clear();
			_nbMeanValues = 0;
			_B::PaintElementBckGround(dw, _B::GetFrame().GetSize(), style->lmntBackgnd);
			const_GridStepIterator iter = GridStepBegin();
			const_GridStepIterator endIter = GridStepEnd();

			if ( _B::GetStackingPriority() > 0) {
				switch(_B::GetElementPos()) {
					case LEFT_OF_GRAPH:
						PaintAxisLeft(dw, scale, iter, endIter);
						break;
					case BOTTOM_OF_GRAPH:
						PaintAxisBottom(dw, scale, iter, endIter);
						break;
					case TOP_OF_GRAPH:
						PaintAxisTop(dw, scale, iter, endIter);
						break;
					case RIGHT_OF_GRAPH:
						PaintAxisRight(dw, scale, iter, endIter);
						break;
					case FLOAT_OVER_GRAPH:
						break;
				}
			}
			else {
				switch(_B::GetElementPos()){
					case LEFT_OF_GRAPH:
						PaintAxisRight(dw, scale, iter, endIter);
						break;
					case BOTTOM_OF_GRAPH:
						PaintAxisTop(dw, scale, iter, endIter);
						break;
					case TOP_OF_GRAPH:
						PaintAxisBottom(dw, scale, iter, endIter);
						break;
					case RIGHT_OF_GRAPH:
						PaintAxisLeft(dw, scale, iter, endIter);
						break;
					case FLOAT_OVER_GRAPH:
						break;
				}
			}
			_meanTickTextsz = _tmpTextSize;
//			LOG("   SumTickTextsz=" << _meanTickTextsz << "   _nbMeanValues=" << _nbMeanValues << "    ==> mean=" << _meanTickTextsz/_nbMeanValues);
			if ( _nbMeanValues>0.9 ) { _meanTickTextsz /= _nbMeanValues; }
			else                     { _meanTickTextsz.cx = 10; _meanTickTextsz.cy=10; }
		}
	};
}


#endif /* GRIDAXISDRAW_H_ */
