#ifndef __TimingPolicies_H__
#define __TimingPolicies_H__

#include "stdio.h"

#include "HWTiming.h"

/** @brief Contient toutes les classes qui realisent des mesures de temps
 */
namespace TimingPolicies_ns
{
	/**
	 * @brief Class qui mesure simplement une duree
	 */
	class BasicTiming : protected HWTiming
	{
		public:
			typedef struct TimingsDataStruct
			{
				template<class STREAM>
				friend STREAM& operator<<(STREAM& str, const TimingsDataStruct& t)
				{
					str << "Duree="<< t.duree << "ms";
					return str;
				}
				double duree;
			} TimingsDataStruct;

		protected:
			BasicTiming(void);

		public:
			~BasicTiming(void);

			inline void beginTiming(void)
			{
				m_Startcount = getTime();
			};

			inline void endTiming(void)
			{
				m_Stopcount = getTime();
				m_timings.duree = diff_ms(m_Startcount, m_Stopcount);
			};

			void reset(void) { m_timings.duree = 0; };

			inline double getDuration(void) const { return m_timings.duree; }
			inline const TimingsDataStruct& getTimings(void) const { return m_timings; }
			template <class STREAM> STREAM& printStats(STREAM& str) const { str << m_timings << "\n"; return str; }
		private:
			timeType m_Startcount;
			timeType m_Stopcount;
			TimingsDataStruct m_timings;
	};


	/**
	 * @brief Class qui mesure une duree et memorise les valeurs MIN et MAX
	 */
	class MinMaxTiming : protected HWTiming
	{
		public:
			typedef struct TimingsDataStruct
			{
				template<class STREAM>
				friend STREAM& operator<<(STREAM& str, const TimingsDataStruct& t)
				{
					str << "Max="<< t.max << "ms   Min=" << t.min << "ms   Current=" << t.duree << "ms";
					return str;
				}
				double duree;
				double max;
				double min;
			} TimingsDataStruct;

		protected:
			MinMaxTiming(void);
			~MinMaxTiming(void);

		public:
			inline void beginTiming(void) {	m_Startcount = getTime(); };

			inline void endTiming(void)	{
				m_Stopcount = getTime();
				m_timings.duree = diff_ms(m_Startcount, m_Stopcount);
				if( m_timings.duree > m_timings.max)       m_timings.max = m_timings.duree;
				else if( m_timings.duree < m_timings.min ) m_timings.min = m_timings.duree;
			};

			void reset(void);

			inline double getDuration(void) const { return m_timings.duree; }
			inline const TimingsDataStruct& getTimings(void) const { return m_timings; }
			template <class STREAM> STREAM& printStats(STREAM& str) const { str << m_timings << "\n"; return str; }

		private:
			timeType m_Startcount;
			timeType m_Stopcount;
			TimingsDataStruct m_timings;
	};



	/**
	 * @brief Class qui mesure une duree et calcul une moyenne mobile (comme un FIR)
	 */
	class AverageTiming : protected HWTiming
	{
		public:
			typedef struct TimingsDataStruct {
				template<class STREAM>
				friend STREAM& operator<<(STREAM& str, const TimingsDataStruct& t)
				{
					str << "Average=" << t.somme / double(t.count) << "ms   ( count:" << t.count << " )    Current=" << t.duree << "ms";
					return str;
				}
				

				unsigned int count;
				double somme;
				double duree;
			} TimingsDataStruct;


		protected:
			AverageTiming(void);
			~AverageTiming(void);

		public:
			inline void beginTiming(void) {
				m_Startcount = getTime();
				m_timings.count++;
			};

			inline void endTiming(void) {
				m_Stopcount = getTime();
				m_timings.somme += (m_timings.duree = diff_ms(m_Startcount, m_Stopcount));
			};

			void reset(void);

			inline double getDuration(void) const { return m_timings.duree; }
			inline const TimingsDataStruct& getTimings(void) const { return m_timings; }
			template <class STREAM> STREAM& printStats(STREAM& str) const { str << m_timings << "\n"; return str; }
		private:
			timeType m_Startcount;
			timeType m_Stopcount;
			TimingsDataStruct m_timings;
	};


	/**
	 * @brief Class qui mesure une duree et memorise les valeurs MIN et MAX et Average
	 */
	class MinMaxAverageTiming : protected HWTiming
	{
		public:
			typedef struct TimingsDataStruct
			{
				template<class STREAM>
				friend STREAM& operator<<(STREAM& str, const TimingsDataStruct& t)
				{
					str << "Max="<< t.max << "ms   Min=" << t.min << "ms   Average=" << t.somme / double(t.count) << "ms   ( count:" << t.count << " )   Current="<< t.duree << "ms";
					return str;
				}
				double max;
				double min;
				unsigned int count;
				double somme;
				double duree;
			} TimingsDataStruct;

		protected:
			MinMaxAverageTiming(void);
			~MinMaxAverageTiming(void);

		public:
			inline void beginTiming(void) {
				m_Startcount = getTime();
				m_timings.count++;
			};

			inline void endTiming(void) {
				m_Stopcount = getTime();
				m_timings.duree  = diff_ms(m_Startcount, m_Stopcount);
				m_timings.somme += m_timings.duree;

				if( m_timings.duree  > m_timings.max)       m_timings.max = m_timings.duree;
				else if( m_timings.duree  < m_timings.min ) m_timings.min = m_timings.duree;
			};

			void reset(void);

			inline double getDuration(void) const { return m_timings.duree; }
			inline const TimingsDataStruct& getTimings(void) const { return m_timings; }
			template <class STREAM> STREAM& printStats(STREAM& str) const { str << m_timings << "\n"; return str; }
			
			
		private:
			timeType m_Startcount;
			timeType m_Stopcount;
			TimingsDataStruct m_timings;
	};


	/**
	 * @brief Class qui mesure une duree et memorise les valeurs MIN et MAX et Average
	 */
	template <int HISTO_SIZE=20>
	class HistogramTiming : protected HWTiming
	{
		public:
			typedef struct TimingsDataStruct
			{
				template<class STREAM>
				friend STREAM& operator<<(STREAM& str, const TimingsDataStruct& t)
				{
					str << HISTO_SIZE << " pos    Range: [ " << t.rangeMin_ms << " , " << t.rangeMax_ms << " ]  (count:" << t.count << ")   somme: " << t.somme << "ms";
					for (int c=0; c<HISTO_SIZE; ++c) {
						str << "\n     [ " << t.thresholds[c] << " , " << t.thresholds[c+1] << " ]ms    " << t.histogram[c];
					}
					return str;
				}
			    unsigned int count;
			    unsigned int histogram[HISTO_SIZE];
			    double thresholds[HISTO_SIZE+1];
			    double rangeMax_ms;
			    double rangeMin_ms;
			    double duree;
			    double somme;
    		} TimingsDataStruct;

		protected:
			HistogramTiming(void) { setRange(0,1000); };
			~HistogramTiming(void)  {};

		public:
			inline void beginTiming(void) {
				m_Startcount = getTime();
			}

			inline void endTiming(void) {
				m_timings.count++;
				m_Stopcount = getTime();
				m_timings.duree = diff_ms(m_Startcount, m_Stopcount );
				m_timings.somme += m_timings.duree;
				addToHisto( m_timings.duree );
			}
			
			void autoRange() {
				setRange(m_timings.thresholds[0], m_timings.thresholds[HISTO_SIZE]);
			}
			
			void setRange( double min, double max ) {
				reset();
				m_timings.rangeMin_ms = min;
				m_timings.rangeMax_ms = max;
				double inc = (max-min)/HISTO_SIZE;
				for (int c=0; c<=HISTO_SIZE; ++c) {
					m_timings.thresholds[c] = min;
					min += inc;
				}
			}

			void reset(void) {
				m_timings.somme = 0;
				m_timings.duree = 0;
				m_timings.count = 0;
				for (int c=0; c<HISTO_SIZE; ++c) {
					m_timings.histogram[c] = 0;
				}
			}

			inline double getDuration(void) const { return m_timings.duree; }
			inline const TimingsDataStruct& getTimings(void) const { return m_timings; }
			template <class STREAM> STREAM& printStats(STREAM& str) const { str << m_timings << "\n"; return str; }
			
			void addToHisto( double val )
			{
				int min = 0;
				int max = HISTO_SIZE;
			
				if (m_timings.thresholds[0]>val)          m_timings.thresholds[0] = val;
				if (m_timings.thresholds[HISTO_SIZE]<val) m_timings.thresholds[HISTO_SIZE] = val;
				
				while(min < max)
				{
					int mid = (max + min) >> 1;
					if( m_timings.thresholds[mid] <= val )
						min = mid + 1;
					else
						max = mid;
				}
				++(m_timings.histogram[min-1]);
				
			}
		
			
		private:
			timeType m_Startcount;
			timeType m_Stopcount;
			TimingsDataStruct m_timings;
	};

	/**
	 * @brief Classe de desactivation des fonctions de mesure de temps
	 */
	class NoTiming : protected HWTiming
	{
		protected:
			NoTiming(void) {};
			~NoTiming(void) {};

		public:
			inline void beginTiming(void) {};
			inline void endTiming(void)  {};
			inline void reset(void) {};

			template <class STREAM>
			inline STREAM& printStats(STREAM& str) const { return str; }
			inline double getDuration(void) const { return 0; }
	};

};// fin namespace : TimingPolicies_ns




/**
 * @brief Class de mesure de temps utiles pour les TESTS UNITAIRES
 * @detail Pour la mesure de temps dans une hierachie de classe
 *         utiliser l'heritage 'TimingPolicies_ns::xxx' permettant une meilleure
 *         integration !
 */
template <class POLICY_TYPE = TimingPolicies_ns::BasicTiming>
class NamedTimings : public POLICY_TYPE
{
	public:
		NamedTimings(const char* prefix)
		: m_prefix( prefix )
		{}

#ifdef CORE_H
			Upp::Stream& printStats(Upp::Stream& str) const {
				typename POLICY_TYPE::TimingsDataStruct timings = POLICY_TYPE::getTimings();
				str << m_prefix << " > " << AsString(timings) << "\n";
				return str;
			}
#endif

		template <class STREAM>
		STREAM& printStats(STREAM& str)	const {
			// faire une copie des donnees avant affichage
			// pour ne pas avoir de PB avec multithread
			// ( en toute rigueur il faudrait un mutex ...,  mais on fait juste de l'affichage ! )
			typename POLICY_TYPE::TimingsDataStruct timings = POLICY_TYPE::getTimings();
			str << m_prefix << " > " << timings << "\n";
			return str;
		}


	private:
		const char* m_prefix;
};


template <>
class NamedTimings<TimingPolicies_ns::NoTiming> : public TimingPolicies_ns::NoTiming
{
	public:
		NamedTimings(const char* prefix)	{}
		template <class STREAM> STREAM& printStats(STREAM& str) {}
};


#ifdef CORE_H
	namespace Upp {
		template <>	inline String AsString(const TimingPolicies_ns::BasicTiming::TimingsDataStruct& a)         { std::stringstream str; str << a; return String(str.str()); }
		template <>	inline String AsString(const TimingPolicies_ns::MinMaxTiming::TimingsDataStruct& a)        { std::stringstream str; str << a; return String(str.str()); }
		template <>	inline String AsString(const TimingPolicies_ns::AverageTiming::TimingsDataStruct& a)       { std::stringstream str; str << a; return String(str.str()); } 
		template <>	inline String AsString(const TimingPolicies_ns::MinMaxAverageTiming::TimingsDataStruct& a) { std::stringstream str; str << a; return String(str.str()); } 
		template <>	inline String AsString(const TimingPolicies_ns::HistogramTiming<10>::TimingsDataStruct& a) { std::stringstream str; str << a; return String(str.str()); } 
		template <>	inline String AsString(const TimingPolicies_ns::HistogramTiming<20>::TimingsDataStruct& a) { std::stringstream str; str << a; return String(str.str()); } 
	};

#endif


#endif /*__TimingPolicies_H__*/
