#ifndef _GraphDraw_GraphDrawBase_h_
#define _GraphDraw_GraphDrawBase_h_


#ifndef __TimingPolicies_H__
class TimingStub
{
	public:
		TimingStub(const char* prefix) {}
		inline void beginTiming(void) {};
		inline void endTiming(void)  {};
		inline void reset(void) {};
		template <class STREAM> STREAM& printStats(STREAM& str)	{ return str; }
		
		Upp::String  ToString() const {
			return String("Class TimingStub ");
		}
};
#endif


class FastMarkPlot : public MarkPlot {
private:
	template <class T>
	void DoPaint(T& w, const double& scale, const GraphDraw_ns::PointScreen& cp, const double& size, const Color markColor) const
	{
		w.DrawLine(cp.x, cp.y, cp.x, cp.y+1, 1, markColor);
	}

public:
	virtual ~FastMarkPlot() {}

	virtual void Paint(Draw &p
					, const double& scale
					, const GraphDraw_ns::PointScreen& cp
					, const double& size
					, const Color& markColor
					, const double& markBorderWidth
					, const Color& markBorderColor) const
	{
		DoPaint(p, scale, cp, size, markColor);
	}
	virtual void Paint(Painter &p
					, const double& scale
					, const GraphDraw_ns::PointScreen& cp
					, const double& size
					, const Color& markColor
					, const double& markBorderWidth
					, const Color& markBorderColor) const
	{
		DoPaint(p, scale, cp, size, markColor);
	}
};

namespace GraphDraw_ns
{
	
	
	// ------------------------------------------------------------------------------------------------------------------------------------------------------
	//
	//            G r a p h D r a  w    C l a s s e s
	//
	// ------------------------------------------------------------------------------------------------------------------------------------------------------




	class CH_EmptyGraphDraw {
		public:
			struct StyleGD : ChStyle<StyleGD> {
				Value    plotBckgndStyle;
				Value    ctrlBckgndStyle;
				Value    rectSelectStyle;
			};
		
			static const StyleGD& StyleDefault();
			
			CH_EmptyGraphDraw() {
				styleGD = &StyleDefault();
			}
		
		protected:
		const StyleGD *styleGD;

	};
	
	#define MAKE_GRAPHDRAW_FN( STYLE_GD, RETURN_CLASS ) \
		RETURN_CLASS& SetPlotBackgroundStyle(Value c) { STYLE_GD.plotBckgndStyle = c; /*_CtrlBackgroundImage.Clear();*/ return *static_cast<RETURN_CLASS*>(this); }\
		RETURN_CLASS& SetCtrlBackgroundStyle(Value c) { STYLE_GD.ctrlBckgndStyle = c; /*_CtrlBackgroundImage.Clear();*/ return *static_cast<RETURN_CLASS*>(this); }

	// ============================
	// CRTP_EmptyGraphDraw   CLASS
	// ============================
	template<class TYPES, class DERIVED>
	class CRTP_EmptyGraphDraw : public CustomSeriesGroup< DERIVED, SeriesGroup< DERIVED > >, public GraphElementParent, public CoordinateConverterOwner, public CH_EmptyGraphDraw
	{
		public:
		typedef CRTP_EmptyGraphDraw<TYPES, DERIVED> CLASSNAME;
		typedef CustomSeriesGroup< DERIVED, SeriesGroup< DERIVED > >  _B;

		virtual void SetStyleGD(const StyleGD& s) {
			styleGD = &s;
			ClearPlotDrawImg();
			_CtrlBackgroundImage.Clear();
		}

		protected:
		// graph elements draw around the graph
		Vector< GraphElement* >  _drawElements;
		Vector< GraphElement* >  _createdElements; // the elements in this list are owned by GrapDraw
		Vector< CoordinateConverter* > _xConverters;
		Vector< CoordinateConverter* > _yConverters;
		

		RectScreen  _ctrlRect;  // whole graph screen Rect
		RectScreen  _plotRect;
		DrawMode    _drawMode;
		bool        _doFastPaint;

		typename TYPES::TypeBlankElement topMargin;
		typename TYPES::TypeBlankElement bottomMargin;
		typename TYPES::TypeBlankElement leftMargin;
		typename TYPES::TypeBlankElement rightMargin;
		
		UndoStack _undoManager;
		Image     _PlotDrawImage;
		Image     _CtrlBackgroundImage;
		
		RectScreen  _selectRect;

#ifndef __TimingPolicies_H__
		typedef TimingStub TimingType;
#else
		typedef NamedTimings< TimingPolicies_ns::MinMaxAverageTiming > TimingType;		
		typedef NamedTimings< TimingPolicies_ns::NoTiming > NullTimingType;		
		//typedef NamedTimings< TimingPolicies_ns::NoTiming >  TimingType;
		
#endif
		
		TimingType _paintTiming;
		TimingType _fastPaintPlotDataTiming;
		TimingType _totalFullPaintPlotDataTiming;
		TimingType _paintPlotDataGlobalTiming;
		TimingType _paintBackGndTiming;
		TimingType _initBackGndPaintTiming;
		TimingType _paintBackGndTiming_chPaint;
		TimingType _paintBackGndTiming_paintImage;
		TimingType _paintBackGndTiming_copyImage;
		TimingType _preparePlotPointsListTiming;
		TimingType _fullPaintPlotDataTiming;

		
		// helper method
		void AppendElementToRect(GraphElement& element, RectScreen& fromRect, const int scale) const
		{
//			RLOGBLOCK_STR( debugTrace, "CRTP_EmptyGraphDraw::AppendElementToRect()   GraphElem:" << element);
			RectScreen res = fromRect;
			switch(element.GetElementPos()) {
				case TOP_OF_GRAPH:
					res.bottom = res.top+element.GetElementWidth()*scale;
					fromRect.top = res.bottom;
					element.SetFrame(res);
					break;
				case BOTTOM_OF_GRAPH:
					res.top = res.bottom-element.GetElementWidth()*scale;
					fromRect.bottom = res.top;
					element.SetFrame(res);
					break;
				case LEFT_OF_GRAPH:
					res.right = res.left+element.GetElementWidth()*scale;
					fromRect.left = res.right;
					element.SetFrame(res);
					break;
				case RIGHT_OF_GRAPH:
					res.left = res.right-element.GetElementWidth()*scale;
					fromRect.right = res.left;
					element.SetFrame(res);
					break;
				case FLOAT_OVER_GRAPH:
					res = element.GetFloatFrame();
					res = RectScreen( res.TopLeft()*scale, res.Size()*scale );
					element.SetFrame(res);
					break;
			}
//			RLOG_STR( debugTrace, "modified plotRect:(" << fromRect << ")");
//			RLOG_STR( debugTrace, "element:(" << element << ")");
		}

		void ClearPlotDrawImg() {
			_PlotDrawImage.Clear();
		}
		
		void updateSizesOnSide( Vector< GraphElement* >& elementsList, RectScreen& tmpRect, RectScreen& plotRect, const int scale = 1 ) const {
//			RLOGBLOCK_STR( debugTrace, "CRTP_EmptyGraphDraw::updateSizesOnSide(" << plotRect << ")");

			int j = elementsList.GetCount()-1;

			for (; j>=1; --j)
			{
				tmpRect = plotRect;
				if ( elementsList[j]->GetOverlapPrevious() ) {
					elementsList[j]->SetElementWidth(0);
					AppendElementToRect(*(elementsList[j]), tmpRect, scale);
					AppendElementToRect(*(elementsList[j-1]), tmpRect, scale);
					if ( elementsList[j-1]->GetStackingPriority() >=0 ) plotRect = tmpRect;
					elementsList[j]->SetElementWidth( elementsList[j-1]->GetElementWidth());
					elementsList[j]->SetFrame( elementsList[j-1]->GetFrame() );
					--j;
					if (j==0) --j;
				}
				else
				{
					AppendElementToRect(*(elementsList[j]), tmpRect, scale);
					if ( elementsList[j]->GetStackingPriority() >=0 ) plotRect = tmpRect;
				}
			}

			if (j==0) {
				tmpRect = plotRect;
				AppendElementToRect(*(elementsList[j]), tmpRect, scale);
				if ( elementsList[j]->GetStackingPriority() >=0 ) plotRect = tmpRect;
			}
		}

		void updateSizes( const int scale = 1 )
		{
//			RLOGBLOCK_STR( debugTrace, "CRTP_EmptyGraphDraw::updateSizes(" << _plotRect << ")");
			RectScreen svg = _plotRect;
			_plotRect = _ctrlRect;
//			RLOG_STR( debugTrace, "INPUT   plotRect:(" << _plotRect << ")");

			Sort(_drawElements, compareGraphElementPriority);

			Vector< GraphElement* >  topSideElements;
			Vector< GraphElement* >  bottomSideElements;
			Vector< GraphElement* >  leftSideElements;
			Vector< GraphElement* >  rightSideElements;
			Vector< GraphElement* >  floatSideElements;
			for (int j=0; j<_drawElements.GetCount(); ++j) {
				if (!_drawElements[j]->IsHidden()) {
					switch(_drawElements[j]->GetElementPos()) {
						case LEFT_OF_GRAPH:
							leftSideElements.Add(_drawElements[j]);
							break;
						case BOTTOM_OF_GRAPH:
							bottomSideElements.Add(_drawElements[j]);
							break;
						case FLOAT_OVER_GRAPH:
							floatSideElements.Add(_drawElements[j]);
							break;
						case RIGHT_OF_GRAPH:
							rightSideElements.Add(_drawElements[j]);
							break;
						case TOP_OF_GRAPH:
							topSideElements.Add(_drawElements[j]);
							break;
					}
				}
			}

			RectScreen tmpRect = _plotRect;
			
			updateSizesOnSide(topSideElements,   tmpRect, _plotRect, scale);
			updateSizesOnSide(bottomSideElements,tmpRect, _plotRect, scale);
			updateSizesOnSide(leftSideElements,  tmpRect, _plotRect, scale);
			updateSizesOnSide(rightSideElements, tmpRect, _plotRect, scale);
			updateSizesOnSide(floatSideElements, tmpRect, _plotRect, scale);
		
			if (_plotRect != svg) {
				_CtrlBackgroundImage.Clear();
				ClearPlotDrawImg();
			}

			for (int j = 0; j < _drawElements.GetCount(); ++j) {
				if (!_drawElements[j]->IsHidden()) {
					_drawElements[j]->AdjustToPlotRect( _plotRect );
				}
			}

			for (int j = 0; j < _xConverters.GetCount(); ++j) {
				_xConverters[j]->updateScreenSize( 0, _plotRect.GetWidth() );
			}

			for (int j = 0; j < _yConverters.GetCount(); ++j) {
				_yConverters[j]->updateScreenSize( _plotRect.GetHeight(), 0  );
			}
		}

		public:

		bool debugTrace;
		bool debugTraceTimings;

		CRTP_EmptyGraphDraw()
		: _ctrlRect(0,0,100,100)
		, _plotRect(0,0,100,100)
		, _drawMode( MD_DRAW )
		, _doFastPaint(false)
		, _paintTiming("paint()")
		, _fastPaintPlotDataTiming("fastPaintPlotDataTiming()")
		, _totalFullPaintPlotDataTiming("totalFullPaintPlotDataTiming()")
		, _paintPlotDataGlobalTiming("paintPlotDataGlobalTiming()")
		, _paintBackGndTiming("paintBckGnd()")
		, _initBackGndPaintTiming("initBackGndPaint()")
		, _paintBackGndTiming_chPaint("paintBackGndTiming_chPaint()")
		, _paintBackGndTiming_paintImage("paintBackGndTiming_paintImage()")
		, _paintBackGndTiming_copyImage("paintBackGndTiming_copyImage()")
		, _preparePlotPointsListTiming("_preparePaintPlotPointsListTiming()")
		, _fullPaintPlotDataTiming("_fullPaintPlotDataTiming()")
		
		, debugTrace(false)
		, debugTraceTimings(false)
		{
			SetStyleGD( StyleDefault() );
			_selectRect.Clear();
			setScreenSize( Size(100,100) ); // set default size
			
			topMargin.SetName("Top Margin");
			bottomMargin.SetName("Bottom Margin");
			leftMargin.SetName("Left Margin");
			rightMargin.SetName("Right Margin");

			AddElement(TOP_OF_GRAPH,    3, topMargin,    INT_MAX);
			AddElement(BOTTOM_OF_GRAPH, 3, bottomMargin, INT_MAX);
			AddElement(RIGHT_OF_GRAPH,  3, rightMargin,  INT_MAX);
			AddElement(LEFT_OF_GRAPH,   3, leftMargin,   INT_MAX);
		};


		virtual ~CRTP_EmptyGraphDraw() {
			for (int j = 0; j < _createdElements.GetCount(); j++)
			{
				delete ( _createdElements[j] );
			}
			_createdElements.Clear();
			
			PrintTimingStats(false);
		}


#ifdef flagGD_TIMINGS
		void PrintTimingStats(bool forcePrint=true) {
			if (forcePrint || debugTraceTimings) {
				VppLog() << "============== TIMING STATS ==================\n";
				_paintTiming.printStats( VppLog() );
				_paintBackGndTiming.printStats(VppLog() );
				_initBackGndPaintTiming.printStats( VppLog() );
				_paintBackGndTiming_chPaint.printStats( VppLog() );
				_paintBackGndTiming_copyImage.printStats( VppLog() );
				_paintBackGndTiming_paintImage.printStats( VppLog() );
				_preparePlotPointsListTiming.printStats( VppLog() );
				_fullPaintPlotDataTiming.printStats( VppLog() );
				_fastPaintPlotDataTiming.printStats( VppLog() );
				_totalFullPaintPlotDataTiming.printStats( VppLog() );
				_paintPlotDataGlobalTiming.printStats( VppLog() );
				VppLog() << "===============================================\n\n";
			}
		}

		void ResetTimingStats() {
			_paintTiming.reset();
			_paintBackGndTiming.reset();
			_initBackGndPaintTiming.reset();
			_paintBackGndTiming_chPaint.reset();
			_paintBackGndTiming_copyImage.reset();
			_paintBackGndTiming_paintImage.reset();
			_fastPaintPlotDataTiming.reset();
			_totalFullPaintPlotDataTiming.reset();
			_fullPaintPlotDataTiming.reset();
			_paintPlotDataGlobalTiming.reset();
			_preparePlotPointsListTiming.reset();
		}
		
		private:
		void DoDataDrawPerfTest2(unsigned int refreshCount ) {
			if (refreshCount) {
				--refreshCount;
				RefreshFromChild(REFRESH_FULL);
				PostCallback(THISBACK1(DoDataDrawPerfTest2, refreshCount));
			}
			else PostCallback(THISBACK1(PrintTimingStats, true));
		}
		public:
		void DoDataDrawPerfTest(unsigned int nbDraws = 10 ) {
			ResetTimingStats();
			PostCallback(THISBACK1(DoDataDrawPerfTest2, nbDraws));
		}

#else
		inline void PrintTimingStats(bool forcePrint=true) {}
		inline void ResetTimingStats() {}
		inline void DoDataDrawPerfTest(unsigned int nbDraws = 10 ) {}
#endif
		
		virtual void SetGDModify() {}
		
		virtual TypeVectorSeries& GetSeries() {
			return _B::series;
		}

		virtual Value GetParentCtrl() {
			ASSERT_(0, "CRTP_EmptyGraphDraw::GetParentCtrl()  was CALLED");
			return Null;
		}

		virtual void AddUndoAction(UndoStackData& CB) {
			_undoManager.AddUndoAction(CB);
		}
		
		void Undo() {
			if (_undoManager.Undo()) RefreshFromChild( GraphDraw_ns::REFRESH_FULL );
		}

		void Redo() {
			if (_undoManager.Redo()) RefreshFromChild( GraphDraw_ns::REFRESH_FULL );
		}
		
		void FitToData(FitToDataStrategy fitStrategy=ALL_SERIES) {
			for (int j = _drawElements.GetCount()-1; j>=0; j--) {
				_drawElements[j]->FitToData(fitStrategy);
			}
			RefreshFromChild( GraphDraw_ns::REFRESH_FULL );
			UpdateAllLinks();
		}
		

		CoordinateConverter& GetXCoordConverter(int c) {
			if (c<0) {
				ASSERT(_B::_currentXConverter!=0);
				return *_B::_currentXConverter;
			}
			CoordinateConverter& coord = *_xConverters[c];
			return (coord);
		}
		
		CoordinateConverter& GetYCoordConverter(int c) {
			if (c <0 ) {
				ASSERT(_B::_currentYConverter!=0);
				return *_B::_currentYConverter;
			}
			CoordinateConverter& coord = *_yConverters[c];
			return (coord);
		}
// ====================== LINKING ======================

	virtual void RefreshWhenLinkUpdate() {
		//Refresh(); //GraphDraw_ns::REFRESH_FAST);
		RefreshFromChild(GraphDraw_ns::REFRESH_FULL);
	}

	virtual void RequestLinksUpdate(CoordinateConverter& p) {
		p.UpdateLinks(LINK_UPDATE_RANGE);
	}

	void UpdateAllLinks(LinkUpdateStrategy strategy = LINK_UPDATE_RANGE) {
		for (int j = 0; j < _xConverters.GetCount(); j++) {
			_xConverters[j]->UpdateLinks(strategy);
		}

		for (int j = 0; j < _yConverters.GetCount(); j++) {
			_yConverters[j]->UpdateLinks(strategy);
		}
	}

	void ForceUpdateAllLinks(LinkUpdateStrategy strategy = LINK_UPDATE_RANGE) {
		for (int j = 0; j < _xConverters.GetCount(); j++) {
			_xConverters[j]->linkStatusCounter.Inc();
		}

		for (int j = 0; j < _yConverters.GetCount(); j++) {
			_yConverters[j]->linkStatusCounter.Inc();
		}
		
		UpdateAllLinks(strategy);
	}

	static void Link(CoordinateConverter& local, CoordinateConverter& other) {
		if (local.Link(other) ) {
			other.Link(local);
		}
	}
	
	static void Unlink(CoordinateConverter& local, CoordinateConverter& other) {
		local.Unlink(other);
	}
// ======================




		virtual DERIVED& GD_DEPRECATED("A deplacer") SetPlotBackgroundStyle(Value c) { /*plotBckgndStyle = c; _CtrlBackgroundImage.Clear();*/ return *static_cast<DERIVED*>(this); }
		virtual DERIVED& GD_DEPRECATED("A deplacer") SetCtrlBackgroundStyle(Value c) { /*_ctrlBckgndStyle = c; _CtrlBackgroundImage.Clear();*/ return *static_cast<DERIVED*>(this); }
		
		DERIVED& SetDrawMode(DrawMode m) { _drawMode = m; return *static_cast<DERIVED*>(this); }
		DERIVED& SetDrawMode(int m) {
			if ((MD_DRAW<=m) && (m<=MD_SUBPIXEL)) _drawMode = (DrawMode)m;
			return *static_cast<DERIVED*>(this);
		}
		virtual DrawMode GetDrawMode() { return _drawMode; }

		
		DERIVED& SetTopMargin(int v)    { topMargin.SetElementWidth(v);    return *static_cast<DERIVED*>(this); }
		DERIVED& SetBottomMargin(int v) { bottomMargin.SetElementWidth(v); return *static_cast<DERIVED*>(this); }
		DERIVED& SetLeftMargin(int v)   { leftMargin.SetElementWidth(v);   return *static_cast<DERIVED*>(this); }
		DERIVED& SetRightMargin(int v)  { rightMargin.SetElementWidth(v);  return *static_cast<DERIVED*>(this); }

		DERIVED& setScreenSize(RectScreen r, const int scale=1)	{
//			RLOGBLOCK_STR( debugTrace, "setScreenSize(" << r << " , " << scale << ")");
			if (r!=_ctrlRect || scale != 1) {
				_ctrlRect = r;
				ClearPlotDrawImg();
				_CtrlBackgroundImage.Clear();
				updateSizes(scale);
			}
			return *static_cast<DERIVED*>(this);
		}

		inline DERIVED& setScreenSize( const int scale=1 )	{
//			RLOGBLOCK_STR( debugTrace, "setScreenSize()");
			return setScreenSize(_ctrlRect, scale);
		}

		virtual void AddXConverter(CoordinateConverter* conv) {
			_xConverters.Add() = conv;
			_B::_currentXConverter = conv;
		}

		virtual void AddYConverter(CoordinateConverter* conv) {
			_yConverters.Add() = conv;
			_B::_currentYConverter = conv;
		}

		template <class COORDCONV>
		COORDCONV& AddXConverter(COORDCONV& conv) {
			AddXConverter(&conv);
			return conv;
		}

		template <class COORDCONV>
		COORDCONV& AddYConverter(COORDCONV& conv) {
			AddYConverter(&conv);
			return conv;
		}
		
		void SetCurrentXConverter(int n) {
			ASSERT( n < _xConverters.GetCount() );
			_B::_currentXConverter =  _xConverters[n];
		}

		void SetCurrentYConverter(int n) {
			ASSERT( n < _yConverters.GetCount() );
			_B::_currentYConverter =  _yConverters[n];
		}

/*
		GD_DEPRECATED("A deplacer dans GraphCtrl") DERIVED&  SetSelectStyle(Value p) {
//			styleGD->rectSelectStyle = p;
			return *static_cast<DERIVED*>(this);
		}
*/

		bool IsValidForZoom(RectScreen r)
		{
			static const TypeGraphCoord rangeMin = std::numeric_limits<TypeGraphCoord>::epsilon() * 1000.0;
			for (int j = 0; j < _xConverters.GetCount(); j++)
			{
				if ( tabs(_xConverters[j]->toGraph( r.left ) - _xConverters[j]->toGraph( r.right )) < rangeMin ) return false;
			}

			for (int j = 0; j < _yConverters.GetCount(); j++)
			{
				if ( tabs(_yConverters[j]->toGraph( r.bottom ) - _yConverters[j]->toGraph( r.top )) < rangeMin) return false;
			}
			return true;
		}

		virtual Callback MakeRestoreGraphSizeCB() {
			Callback action;
			for (int j = 0; j < _xConverters.GetCount(); j++) { action << _xConverters[j]->MakeRestoreAxisMinMaxCB(); }
			for (int j = 0; j < _yConverters.GetCount(); j++) { action << _yConverters[j]->MakeRestoreAxisMinMaxCB(); }
			return 	action;
		}

		virtual void ZoomX(TypeScreenCoord left, TypeScreenCoord right)
		{
			bool isModified = false;
			for (int j = 0; j < _xConverters.GetCount(); j++)
			{
				if (_xConverters[j]->IsZoomAllowed()) {
					_xConverters[j]->updateGraphSize( _xConverters[j]->toGraph( left ),
					                                  _xConverters[j]->toGraph( right ));
					_xConverters[j]->linkStatusCounter.Inc();
					isModified = true;
				}
			}
			if (isModified) {
				ClearPlotDrawImg();
				Refresh();
			}
		}

		virtual void ZoomY(TypeScreenCoord top, TypeScreenCoord bottom)
		{
			bool isModified = false;
			for (int j = 0; j < _yConverters.GetCount(); j++)
			{
				if (_yConverters[j]->IsZoomAllowed()) {
					_yConverters[j]->updateGraphSize( _yConverters[j]->toGraph( bottom ),
					                                  _yConverters[j]->toGraph( top ));
					_yConverters[j]->linkStatusCounter.Inc();
					isModified = true;
				}
			}
			if (isModified) {
				ClearPlotDrawImg();
				Refresh();
			}
		}

		void ZoomOnRect(RectScreen r)
		{
			if ( IsValidForZoom(r) )
			{
				ZoomX(r.left, r.right);
				ZoomY(r.top, r.bottom);
				Refresh();
			}
		}

		void ApplyInvZoomFactor( TypeGraphCoord zoom)
		{
			ApplyZoomFactor( 1.0/zoom);
		}

		void ApplyZoomFactor( TypeGraphCoord zoom)
		{
			TypeScreenCoord xDelta = (TypeScreenCoord)(_plotRect.GetWidth() * (1.0 - zoom) + .5);
			TypeScreenCoord yDelta = (TypeScreenCoord)(_plotRect.GetHeight() * (1.0 - zoom) + .5);
			Rect r( -xDelta/2,
					-yDelta/2,
					_plotRect.GetWidth() +xDelta - xDelta/2,
					_plotRect.GetHeight() +yDelta - yDelta/2 );
			ZoomOnRect(r);
		}

		virtual void ScrollX( TypeScreenCoord xOffset, bool doRefre = true)
		{
			if (xOffset == 0) return;
			bool isModified = false;
			for (int j = 0; j < _xConverters.GetCount(); j++)
			{
				if (_xConverters[j]->IsScrollAllowed()) {
					_xConverters[j]->Scroll( xOffset );
					_xConverters[j]->linkStatusCounter.Inc();
					isModified=true;
				}
			}
			if (isModified) {
				ClearPlotDrawImg();
				if (doRefre) Refresh();
			}
		}

		virtual void ScrollY( TypeScreenCoord yOffset, bool doRefre = true)
		{
			if (yOffset == 0) return;
			bool isModified = false;
			for (int j = 0; j < _yConverters.GetCount(); j++)
			{
				if (_yConverters[j]->IsScrollAllowed()) {
					_yConverters[j]->Scroll( yOffset );
					_yConverters[j]->linkStatusCounter.Inc();
					isModified=true;
				}
			}
			if (isModified) {
				ClearPlotDrawImg();
				if (doRefre) Refresh();
			}
		}

		DERIVED& Scroll( TypeScreenCoord xOffset, TypeScreenCoord yOffset)
		{
			ScrollX(xOffset, false);
			ScrollY(yOffset, false);
			Refresh();
			return *static_cast<DERIVED*>(this);
		}

		template<class T, int POS_OF_GRAPH>
		T& AddElement(T& v, int stackPrio) {
			v._parent = this;
			v.SetStackingPriority(stackPrio);
			v.SetElementPos((ElementPosition)POS_OF_GRAPH);
			_drawElements.Add(&v);
			Sort(_drawElements, compareGraphElementPriority);
			return v;
		}

		template<class T>
		T& AddElement(ElementPosition pos, T& v, int stackPrio) {
			v._parent = this;
			v.SetStackingPriority(stackPrio);
			v.SetElementPos(pos);
			_drawElements.Add(&v);
			Sort(_drawElements, compareGraphElementPriority);
			return v;
		}

		template<class T, int POS_OF_GRAPH>
		T& AddElement(int elementWidth, T& v, int stackPrio) {
				v.SetElementWidth(elementWidth);
				return AddElement<T, POS_OF_GRAPH>(v, stackPrio);
		}

		template<class T>
		T& AddElement(ElementPosition pos, int elementWidth, T& v, int stackPrio) {
				v.SetElementWidth(elementWidth);
				return AddElement<T>(pos, v, stackPrio);
		}

		template<class T>  T& AddLeftElement(T& v, int stackPrio)   { return AddElement<T, LEFT_OF_GRAPH>(v, stackPrio); }
		template<class T>  T& AddRightElement(T& v, int stackPrio)  { return AddElement<T, RIGHT_OF_GRAPH>(v, stackPrio); }
		template<class T>  T& AddTopElement(T& v, int stackPrio)    { return AddElement<T, TOP_OF_GRAPH>(v, stackPrio); }
		template<class T>  T& AddBottomElement(T& v, int stackPrio) { return AddElement<T, BOTTOM_OF_GRAPH>(v, stackPrio); }
		template<class T>  T& AddFloatElement(T& v, int stackPrio)  { return AddElement<T, FLOAT_OVER_GRAPH>(v, stackPrio); }

		template<class T>  T& AddLeftElement(int elementWidth, T& v, int stackPrio)   { return AddElement<T, LEFT_OF_GRAPH>(elementWidth, v, stackPrio); }
		template<class T>  T& AddRightElement(int elementWidth, T& v, int stackPrio)  { return AddElement<T, RIGHT_OF_GRAPH>(elementWidth, v, stackPrio); }
		template<class T>  T& AddTopElement(int elementWidth, T& v, int stackPrio)    { return AddElement<T, TOP_OF_GRAPH>(elementWidth, v, stackPrio); }
		template<class T>  T& AddBottomElement(int elementWidth, T& v, int stackPrio) { return AddElement<T, BOTTOM_OF_GRAPH>(elementWidth, v, stackPrio); }
		template<class T>  T& AddFloatElement(int elementWidth, T& v, int stackPrio)   { return AddElement<T, FLOAT_OVER_GRAPH>(elementWidth, v, stackPrio); }

		template<class T, int POS_OF_GRAPH>
		T& CreateElement(int elementWidth, int stackPrio) {
			T* e = new T();
			e->SetElementWidth(elementWidth);
			e->_parent = this;
			_createdElements << e; // to manage object destruction
			AddElement<T, POS_OF_GRAPH>(*e, stackPrio);
			return *e;
		}

		template<class T>
		T& CreateElement(ElementPosition pos, int elementWidth, int stackPrio) {
			T* e = new T();
			e->SetElementWidth(elementWidth);
			e->_parent = this;
			_createdElements << e; // to manage object destruction
			AddElement<T>(pos, *e, stackPrio);
			return *e;
		}

		template<class T, int POS_OF_GRAPH, class P1>
		T& CreateElement1(int elementWidth, int stackPrio, P1& p1 ) {
			T* e = new T(p1);
			e->SetElementWidth(elementWidth);
			e->_parent = this;
			_createdElements << e; // to manage object destruction
			AddElement<T, POS_OF_GRAPH>(*e, stackPrio);
			return *e;
		}

		template<class T, int POS_OF_GRAPH, class P1, class P2>
		T& CreateElement2(int elementWidth, int stackPrio, P1& p1, P2& p2 ) {
			T* e = new T(p1,p2);
			e->SetElementWidth(elementWidth);
			e->_parent = this;
			_createdElements << e; // to manage object destruction
			AddElement<T, POS_OF_GRAPH>(*e, stackPrio);
			return *e;
		}

		void AddBlanklArea(ElementPosition pos, int width, int stackPrio) {
			CreateElement<typename TYPES::TypeBlankElement>( pos, width, stackPrio );
		}
		

		// Refresh called from child
		virtual void RefreshFromChild( RefreshStrategy doFastPaint ) {
			Refresh();
		};

		virtual void Refresh() {};


		Image GetImage( Size size, Color backGndColor = Upp::White(), const int scale = 1 ) {
//			RLOGBLOCK_STR( debugTrace, "CRTP_EmptyGraphDraw::GetImage(" << this << ")   [ FastPaint , PlotImgEmpty ] => [ " << _doFastPaint << " , " << _PlotDrawImage.IsEmpty() << " ]");
			RectScreen screenRectSvg = _ctrlRect;
			Image plotDrawImageSvg = _PlotDrawImage;
			Image ctrlBackgroundImageSvg = _CtrlBackgroundImage;
			setScreenSize( size, scale );
			ImageBuffer ib(size);
			Upp::Fill( ib.Begin(),backGndColor, ib.GetLength() );
			BufferPainter bp(ib, MD_ANTIALIASED); 
			ClearPlotDrawImg();
			Paint(bp, scale);
			ClearPlotDrawImg();
			setScreenSize( screenRectSvg );
			_PlotDrawImage = plotDrawImageSvg;
			_CtrlBackgroundImage = ctrlBackgroundImageSvg;
			return ib;
		}

		Image GetImage(Size size, const int scale = 1 ) {
			return GetImage(size, White(), scale );
		}

		inline Image GetImage(const int scale=1) {
			return GetImage( _ctrlRect.Size()*scale, scale );
		}

		inline Image GetImage(Color backGndColor, const int scale=1) {
			return GetImage( _ctrlRect.Size()*scale, backGndColor, scale );
		}



		#define addToFullPointsList_withLine(testFct, pointList, xx, yy , xConverter, yConverter, nbVisiblePoints, prevPoint, prevPointIsAdded, isSeriesFilled, rect) {\
			PointGraph currPoint(xx, yy);\
			if (isSeriesFilled && xConverter.IsInGraphVisibleRange(xx) )\
			{\
				if (!prevPointIsAdded) { pointList.AddPick(PointScreen(xConverter.toScreen( prevPoint.x ), yConverter.toScreen( prevPoint.y ))); }\
				pointList.AddPick(PointScreen(xConverter.toScreen( xx ), yConverter.toScreen( yy )));\
				prevPointIsAdded=true;\
			}\
			else {\
				if ( testFct(prevPoint, currPoint, rect) )\
				{\
					++nbVisiblePoints;\
					if (!prevPointIsAdded) { pointList.AddPick(PointScreen(xConverter.toScreen( prevPoint.x ), yConverter.toScreen( prevPoint.y ))); }\
					pointList.AddPick(PointScreen(xConverter.toScreen( xx ), yConverter.toScreen( yy )));\
					prevPointIsAdded=true;\
				} else { \
					prevPointIsAdded = false;\
				}\
				prevPoint = currPoint;\
			}\
		}


		template<class T>
		void PaintPlotData(T& dw, int scale)
		{
			if ( _doFastPaint ) _fastPaintPlotDataTiming.beginTiming();
			else                _totalFullPaintPlotDataTiming.beginTiming();
			
			if (!_B::series.IsEmpty())
			{
				Vector< PointGraph > p1;
				
				for ( int j = 0; j < _B::series.GetCount(); j++)
				{
					SeriesConfig& currSerie = _B::series[j];
					DataSource* dataSrc = currSerie.PointsData();
					CoordinateConverter& xConverter = *(currSerie.xConverter);
					CoordinateConverter& yConverter = *(currSerie.yConverter);

					if ((currSerie.show==false) || (!currSerie.seriesPlot && !currSerie.markPlot))
						continue;


					if ((currSerie.nbVisiblePoints==0)  &&  ( ! dataSrc->IsExplicit())) {
							currSerie.nbVisiblePoints = (decltype(currSerie.nbVisiblePoints))(dataSrc->GetCount());
					}
					
					// ============================================
					//     CREATE  LIST  OF  POINTS  TO  DRAW
					// ============================================
					int inc = 1;
					Upp::int64 nbVisiblePoints = 0;
					int imin=0, imax=0;
					double dmin=0, dmax=0;
					
					// =================================
					// figure out data-range
					Rectf logicalRect(xConverter.getGraphMin(), yConverter.getGraphMin(), xConverter.getGraphMax(), yConverter.getGraphMax()); 
					if (dataSrc->IsParam()) {             // It is a param function : x=F(t)  y=G(t)
						imin = 0;
						imax = (decltype(imax)) (dataSrc->GetCount()-1);
					} else if ( dataSrc->IsExplicit() ) { // It is a function :  y=f(x)
						dmin = xConverter.getGraphMin(); //ffloor(xConverter.getGraphMin() - 1);
						dmax = xConverter.getGraphMax(); //ffloor(xConverter.getGraphMax() + 2);
					} else {
					    imin = 0;
					    imax = (decltype(imax)) (dataSrc->GetCount()-1);
					    if (currSerie.sequential) {
							imin = FindIndexX(dataSrc, imin, imax, xConverter.getGraphMin())-1;
							if (imin < 0) imin=0;
							
							int imax2 = FindIndexX(dataSrc, imin, imax, xConverter.getGraphMax())+1;
							if (imax2>imax) imax2 = imax;
							imax = imax2;
						}
					}
					// =================================
					
					p1.Trim(0);
					p1.Reserve(imax-imin+1);
					PointGraph prevPoint = Null;
					bool prevPointIsVisible = true;
					TypeGraphCoord x;
					TypeGraphCoord y;

					if ( !_doFastPaint )  // DRAW ALL DATA (no partial draw)
					{
						_preparePlotPointsListTiming.beginTiming();
						nbVisiblePoints = 0;
						bool isSeriesFilled = false;
						if ( !currSerie.seriesPlot.IsEmpty() && !currSerie.fillColor.IsNullInstance() ) {
							isSeriesFilled = true;
						}

						if (dataSrc->IsParam())
						{
							const int xmax = imax+1;
							if (!currSerie.seriesPlot.IsEmpty()) // lines & points will be drawn
							{
								for (double cx=imin; cx<=xmax; ++cx)
								{
									x = dataSrc->x(cx);
									y = dataSrc->y(cx);
									p1.AddPick(PointScreen(xConverter.toScreen( x ), yConverter.toScreen( y )));
									++nbVisiblePoints;
								}
							}
							else  // only points will be drawn
							{
								for (double cx=imin; cx<xmax; ++cx)
								{
									x = dataSrc->x(cx);
									y = dataSrc->y(cx);
									if ( logicalRect.Contains(x,y) ) {
										p1.AddPick(PointScreen(xConverter.toScreen( x ), yConverter.toScreen( y )));
										++nbVisiblePoints;
									}
								}
							}
						}
						else if ( dataSrc->IsExplicit() )
						{
							double dx = double(dmax - dmin)/(double)xConverter.getScreenRange(); // one point per X pixel
							dmax = dmax+dx;
							double yy, xx;
							if (!currSerie.seriesPlot.IsEmpty()) // lines & points will be drawn
							{
								for (xx = dmin-dx; xx < dmax; xx += dx)
								{
									yy = dataSrc->f(xx);
									p1.AddPick(PointScreen(xConverter.toScreen( xx ), yConverter.toScreen( yy )));
									++nbVisiblePoints;
								}
							}
							else // only points will be drawn
							{
								for ( xx = dmin-dx; xx < dmax; xx += dx) 
								{
									yy = dataSrc->f(xx);
									if ( yConverter.IsInGraphVisibleRange(yy) )	{
										p1.AddPick(PointScreen(xConverter.toScreen( xx ), yConverter.toScreen( yy )));
										++nbVisiblePoints;
									}
								}
							}
						}
						else
						{
							if (currSerie.sequential)
							{
								int64 c=imin;
								if (!currSerie.seriesPlot.IsEmpty())  // lines & points will be drawn
								{
									prevPoint.x = dataSrc->x(c);
									prevPoint.y = dataSrc->y(c);
									for ( ; c<=imax; ++c)
									{
										x = dataSrc->x(c);
										y = dataSrc->y(c);
										addToFullPointsList_withLine( DetectSCrossing_XinRange, p1, x, y, xConverter, yConverter, nbVisiblePoints, prevPoint, prevPointIsVisible, isSeriesFilled, logicalRect);
									}
								}
								else // only points will be drawn
								{
									for ( ; c<=imax; ++c)
									{
										x = dataSrc->x(c);
										y = dataSrc->y(c);
										if ( yConverter.IsInGraphVisibleRange(y) )	{
											p1.AddPick(PointScreen(xConverter.toScreen( x ), yConverter.toScreen( y )));
											++nbVisiblePoints;
										}
									}
								}
							}
							else
							{
								if (!currSerie.seriesPlot.IsEmpty()) // lines & points will be drawn
								{
									for (Upp::int64 c=imin; c<=imax; c+=inc)
									{
										x = dataSrc->x(c);
										y = dataSrc->y(c);
										addToFullPointsList_withLine( DetectSCrossing, p1, x, y, xConverter, yConverter, nbVisiblePoints, prevPoint, prevPointIsVisible, isSeriesFilled, logicalRect);
									}
								}
								else // only points will be drawn
								{
									for (Upp::int64 c=imin; c<=imax; c+=inc)
									{
										x = dataSrc->x(c);
										y = dataSrc->y(c);
										if ( logicalRect.Contains(x,y) ) {
											p1.AddPick(PointScreen(xConverter.toScreen( x ), yConverter.toScreen( y )));
											++nbVisiblePoints;
										}
									}
								}
							}
						}
						currSerie.nbVisiblePoints = (decltype(currSerie.nbVisiblePoints)) nbVisiblePoints;
						_preparePlotPointsListTiming.endTiming();
						_fullPaintPlotDataTiming.beginTiming();
					}
					else  // DO FAST DRAW : display 800 points maximum per series
					{
						if (dataSrc->IsParam()) {
							const double xmax = imax+1;
							double x, y;
							double dx = Upp::max(double(xmax - imin)/800. , 1.0);
							for (double cx=imin; cx<=xmax; cx+=dx)
							{
								x = dataSrc->x(cx);
								y = dataSrc->y(cx);
								p1 << PointScreen(xConverter.toScreen( x ), yConverter.toScreen( y ));
							}
						} else if (dataSrc->IsExplicit() ) {
							double dx = double(dmax - dmin)/800.;
							dmax = dmax+dx;
							if (xConverter.getScreenRange()<800.) dx = double(dmax - dmin)/xConverter.getScreenRange();

							for (double xx = dmin; xx < dmax; xx += dx) {
								double yy = dataSrc->f(xx);
								p1 << PointScreen(xConverter.toScreen( xx ), yConverter.toScreen( yy ));
							}
						} else {
							inc = fceil((imax-imin)/800.0);
							if (inc==0) ++inc;
							
							imin -= imin%inc;
							
							for ( Upp::int64 c=imin; c<=imax; c+=inc)
							{
								p1 << PointScreen(xConverter.toScreen( dataSrc->x(c)),
								            yConverter.toScreen( dataSrc->y(c)));
							}
						}
					}

					// Draw lines
					if ( !currSerie.seriesPlot.IsEmpty() && (p1.GetCount()>0))
						currSerie.seriesPlot->Paint(dw,
						                                 p1,
						                                 scale,
						                                 currSerie.opacity,
						                                 fround(currSerie.thickness),
						                                 currSerie.color,
						                                 currSerie.dash,
						                                 Null, //style->plotBckgndStyle,
						                                 currSerie.fillColor,
						                                 xConverter.getScreenRange() / xConverter.getGraphRange(),
						                                 yConverter.getScreenRange() / yConverter.getGraphRange(),
						                                 int(yConverter.getScreenRange() * (1 + yConverter.getGraphMin() / yConverter.getGraphRange())),
						                                 10,
						                                 false
						);
					// Draw marks
					if (currSerie.markWidth >= 1 && currSerie.markPlot && !(currSerie.markColor.IsNullInstance()) )
					{
						if ( !currSerie.markPlot.IsEmpty() ) {
							for (int c=0; c<p1.GetCount(); ++c)
							{
								currSerie.markPlot->Paint( dw,
							                               scale,
							                               p1[c],
							                               currSerie.markWidth,
							                               currSerie.markColor,
							                               currSerie.markWidth,
							                               currSerie.markColor);
							}
						}
					}
					
					if ( !_doFastPaint ) _fullPaintPlotDataTiming.endTiming();

				}
			}
			
			if ( _doFastPaint ) _fastPaintPlotDataTiming.endTiming();
			else                _totalFullPaintPlotDataTiming.endTiming();
		}

		template<class T>
		void Paint(T& dw, int scale)
		{
//			RLOGBLOCK_STR( debugTrace, "CRTP_EmptyGraphDraw::Paint(" << this << ")   [ FastPaint , PlotImgEmpty ] => [ " << _doFastPaint << " , " << _PlotDrawImage.IsEmpty() << " ] ===========================");
			_paintTiming.beginTiming();
			_paintBackGndTiming.beginTiming();

			if ( _CtrlBackgroundImage.IsEmpty() )
			{
				_initBackGndPaintTiming.beginTiming();
				RGBA bckgColor;   bckgColor.r = 0; bckgColor.g = 0; bckgColor.b = 0; bckgColor.a = 0;
				ImageBuffer ib(_ctrlRect.Size());
				Upp::Fill( ib.Begin(), bckgColor, ib.GetLength() );
				BufferPainter bp(ib, _drawMode);
				_initBackGndPaintTiming.endTiming();
				// ------------
				// paint background
				// ------------
				_paintBackGndTiming_chPaint.beginTiming();
				if ( !styleGD->ctrlBckgndStyle.IsNull() ) ChPaint(bp, _ctrlRect, styleGD->ctrlBckgndStyle );
				if ( !styleGD->plotBckgndStyle.IsNull())  ChPaint(bp, _plotRect, styleGD->plotBckgndStyle );
				_paintBackGndTiming_chPaint.endTiming();
				_paintBackGndTiming_copyImage.beginTiming();
				_CtrlBackgroundImage = ib;
				_paintBackGndTiming_copyImage.endTiming();
			}
			else
			{
				_initBackGndPaintTiming.reset();
				_paintBackGndTiming_chPaint.reset();
				_paintBackGndTiming_copyImage.reset();
			}
			_paintBackGndTiming_paintImage.beginTiming();
			dw.DrawImage(0, 0, _CtrlBackgroundImage );
			_paintBackGndTiming_paintImage.endTiming();
			
			_paintBackGndTiming.endTiming();

			// --------------------------------------
			// BEGIN paint in PLOT AREA
			// --------------------------------------
			dw.Clipoff(_plotRect.left, _plotRect.top, _plotRect.GetWidth(), _plotRect.GetHeight());

			for (int j = 0; j < _drawElements.GetCount(); j++) {
				if (!_drawElements[j]->IsHidden()) {
					_drawElements[j]->PrePaint(scale);
				}
			}


			// --------------
			// GRAPH ELEMENTS on PLOT area --UNDER DATA-- ( X/Y Grid, or anything else )
			// --------------
			for (int j = 0; j < _drawElements.GetCount(); j++) {
				if ( (!_drawElements[j]->IsHidden()) && _drawElements[j]->IsFloat() ) _drawElements[j]->PaintOnPlot_underData(dw, _plotRect.GetWidth(), scale);
			}
			
			for (int j = 0; j < _drawElements.GetCount(); j++) {
				if (!_drawElements[j]->IsHidden()) {
					if      ( _drawElements[j]->IsVertical() )   _drawElements[j]->PaintOnPlot_underData(dw, _plotRect.GetWidth(), scale);
					else if ( _drawElements[j]->IsHorizontal() ) _drawElements[j]->PaintOnPlot_underData(dw, _plotRect.GetHeight(), scale);
				}
			}
			
			// ----------------
			// paint PLOT DATA
			// ----------------
			_paintPlotDataGlobalTiming.beginTiming();
			
			if ( ( _plotRect.GetHeight() > 0 ) && ( _plotRect.GetWidth() > 0 ) ) {
				if (_doFastPaint)
				{
					RGBA bckgColor;   bckgColor.r = 0; bckgColor.g = 0; bckgColor.b = 0; bckgColor.a = 0;
					ImageBuffer ib(_plotRect.Size());
					Upp::Fill( ib.Begin(), bckgColor, ib.GetLength() );
					BufferPainter bp(ib, _drawMode);
					PaintPlotData(bp, scale);
					_B::PaintAllCustomSeries(bp, scale);
					dw.DrawImage(0, 0, ib);
					ClearPlotDrawImg();
				}
				else
				{
					if ( _B::IsModifyCustomSeries() || _B::IsModifiedData() ) {
						ClearPlotDrawImg();
					}
					if ( _PlotDrawImage.IsEmpty() )
					{
						RGBA bckgColor;   bckgColor.r = 0; bckgColor.g = 0; bckgColor.b = 0; bckgColor.a = 0;
						ImageBuffer ib(_plotRect.Size());
						Upp::Fill( ib.Begin(), bckgColor, ib.GetLength() );
						BufferPainter bp(ib, _drawMode);
						PaintPlotData(bp, scale);
						_B::PaintAllCustomSeries(bp, scale);
						_PlotDrawImage = ib;
						_B::ClearModifyCustomSeries();
						_B::ClearModifyData();
					}
					dw.DrawImage(0, 0, _PlotDrawImage);
				}
			}
			_paintPlotDataGlobalTiming.endTiming();
			
			// --------------
			// GRAPH ELEMENTS on PLOT area --OVER DATA-- ( X/Y Grid, or anything else )
			// --------------
			for (int j = 0; j < _drawElements.GetCount(); j++) {
				if ( (!_drawElements[j]->IsHidden()) && _drawElements[j]->IsFloat() ) _drawElements[j]->PaintOnPlot_overData(dw, _plotRect.GetWidth(), scale);
			}
			
			for (int j = 0; j < _drawElements.GetCount(); j++) {
				if (!_drawElements[j]->IsHidden()) {
					if      ( _drawElements[j]->IsVertical() )   _drawElements[j]->PaintOnPlot_overData(dw, _plotRect.GetWidth(), scale);
					else if ( _drawElements[j]->IsHorizontal() ) _drawElements[j]->PaintOnPlot_overData(dw, _plotRect.GetHeight(), scale);
				}
			}
			
			// --------------------------------------
			// Paint SELECT Rect
			// --------------------------------------
			if ( !_selectRect.IsEmpty() && !styleGD->rectSelectStyle.IsNull()) {
				ChPaint(dw, _selectRect, styleGD->rectSelectStyle);
			}
			// --------------------------------------
			// END of paint in PLOT AREA
			// --------------------------------------
			dw.End();

			// --------------
			// GRAPH ELEMENTS ( painted in they're own area : the ones around the PLOT area )
			// --------------
			
			for (int j = 0; j < _drawElements.GetCount(); j++)
			{
				if ( (!_drawElements[j]->IsHidden()) && (!_drawElements[j]->IsFloat()) ) {
					dw.Offset(_drawElements[j]->GetFrame().TopLeft());
					_drawElements[j]->PaintElement(dw, scale);
					dw.End();
				}
			}
			// --------------
			// GRAPH ELEMENTS on ALL GRAPH area == FLOATING ELEMENTS  ( legend, or anything else )
			// --------------
			for (int j = 0; j < _drawElements.GetCount(); j++)
			{
				if ( (!_drawElements[j]->IsHidden()) && (_drawElements[j]->IsFloat()) ) {
					dw.Clipoff( _drawElements[j]->GetFloatFrame(scale) );
					_drawElements[j]->PaintFloatElement(dw, scale);
					dw.End();
				}
			}
			_paintTiming.endTiming();
			
			//PrintTimingStats(false);
		}
	};
};

template <class T>
class BlurringDecorator : public T {
	
	private:
	bool doBlur;

	void blurImage(Image& m) {
		m = Sharpen(m, -200);
		m = Sharpen(m, -200);
		m = Sharpen(m, -200);
		m = Sharpen(m, -200);
		m = Sharpen(m, -200);
		m = Sharpen(m, -200);
	}


	public:
	
	BlurringDecorator() : doBlur(true) {}
	virtual ~BlurringDecorator() {}
	void setBlur(bool blur = true)  { doBlur = blur; }
	
	virtual void Paint(Draw& w) {
		if (doBlur) {
			ImagePainter dw(T::GetSize(), MODE_NOAA);
			T::Paint(dw);
			Image im = dw;
			blurImage(im);
			w.DrawImage(0, 0, im);
		}
		else {
			T::Paint(w);
		}
	}
};


#endif
