#include "GraphDraw.h"



namespace Upp {
namespace GraphDraw_ns {

	CoordinateConverter::CoordinateConverter(CoordinateConverterOwner& owner)
	: _owner(owner)
	, _isZoomAllowed(true)
	, _isScrollAllowed(true)
	{
		SetConvStd();
		update( 0, 100, 0, 100);
	}
	
	CoordinateConverter::CoordinateConverter(CoordinateConverterOwner& owner, TypeGraphCoord graphMin, TypeGraphCoord graphMax, TypeScreenCoord screenMin, TypeScreenCoord screenMax)
	: _owner(owner)
	, _isZoomAllowed(true)
	, _isScrollAllowed(true)
	{
		SetConvStd();
		update( graphMin, graphMax, screenMin, screenMax);
	}
	
	CoordinateConverter::~CoordinateConverter() {
		UnlinkAll();
	}
	
	void CoordinateConverter::UpdateLinks(LinkUpdateStrategy strategy) {
		for (int l=(links.GetCount()-1); l>=0; --l) {
			if (links[l]->linkStatusCounter != linkStatusCounter) {
				links[l]->linkStatusCounter = linkStatusCounter;
				if (strategy & LINK_UPDATE_LIMITS) {
					links[l]->_graphMinLimit = _graphMinLimit;
					links[l]->_graphMaxLimit = _graphMaxLimit;
					links[l]->Update();
				}
				if (strategy & LINK_UPDATE_RANGE) {
					links[l]->updateGraphSize(_graphMin, _graphMax);
				}
				links[l]->_owner.RefreshWhenLinkUpdate();
				links[l]->UpdateLinks(strategy); // update indirect links
			}
		}
	}
	/*
	void CoordinateConverter::Link(CoordinateConverter& local, CoordinateConverter& other) {
		if (local.Link(other) ) {
			other.Link(local);
		}
	}
	
	void CoordinateConverter::Unlink(CoordinateConverter& local, CoordinateConverter& other) {
		local.Unlink(other);
	}
	*/
	bool CoordinateConverter::Link(CoordinateConverter& conv) {
		if (links.Find(&conv) >= 0) return false;
		links.Add(&conv);
		return true;
	} 
	
	void CoordinateConverter::Unlink(CoordinateConverter& conv) {
		if (links.Find(&conv) < 0) return;
		links.RemoveKey(&conv);
		conv.Unlink(*this);
	} 
	
	void CoordinateConverter::UnlinkAll() {
		while (links.GetCount()>0) {
			Unlink(*(links[0]));
		}
	}
	
	TypeGraphCoord CoordinateConverter::applyRangeLimits(TypeGraphCoord v) {
		if (!_graphMaxLimit.IsNull()) {
			if (v > ValueTo<TypeGraphCoord>(_graphMaxLimit)) v = _graphMaxLimit;
		}
		if (!_graphMinLimit.IsNull()) {
			if (v < ValueTo<TypeGraphCoord>(_graphMinLimit)) v = _graphMinLimit;
		}
		return v;
	}
	
	bool CoordinateConverter::isInRangeLimits(TypeGraphCoord v) {
		if (!_graphMaxLimit.IsNull()) {
			if (v > ValueTo<TypeGraphCoord>(_graphMaxLimit)) return false;;
		}
		if (!_graphMinLimit.IsNull()) {
			if (v < ValueTo<TypeGraphCoord>(_graphMinLimit)) return false;;
		}
		return true;
	}
	
	void CoordinateConverter::update(TypeGraphCoord graphMin, TypeGraphCoord graphMax, TypeScreenCoord screenMin, TypeScreenCoord screenMax, bool doSwap)
	{
		if (doSwap && (graphMin>graphMax)) Swap(graphMin, graphMax);

		if (screenMin!=_screenMin  || screenMax!=_screenMax || _graphMin!=graphMin || _graphMax!=graphMax ) {
			//LOG("CoordinateConverter.update(" << graphMin << ", " << graphMax  << ", " << screenMin<< ", " << screenMax << " )" );
			_graphMin = applyRangeLimits(graphMin);
			_graphMax = applyRangeLimits(graphMax);
			_screenMin = screenMin;
			_screenMax = screenMax;
			_graphRange =  tabs(_graphMax - _graphMin);
			_screenRange = tabs(_screenMax - _screenMin);
	
			_fctGraphMin = _convertFct(_graphMin);
			_fctGraphMax = _convertFct(_graphMax);
			_graphToScreenScale = (decltype(_graphToScreenScale))( (_screenMax - _screenMin) / (_fctGraphMax - _fctGraphMin));
			_screenToGraphScale = (decltype(_screenToGraphScale))( (_fctGraphMax - _fctGraphMin) / (_screenMax - _screenMin));
			_linConv.Update(_fctGraphMin, _fctGraphMax, _screenMin, _screenMax);
			incChangeStatus();
		}
	}
	/*
	void CoordinateConverter::updateScreenSize(TypeScreenCoord screenMin, TypeScreenCoord screenMax)
	{
		if (screenMin!=_screenMin  || screenMax!=_screenMax ) {
			//LOG("CoordinateConverter.update( "<< screenMin<< ", " << screenMax << " )" );
			_screenMin = screenMin;
			_screenMax = screenMax;
			_screenRange = tabs(_screenMax - _screenMin);
	
			_graphToScreenScale = (decltype(_graphToScreenScale))( (_screenMax - _screenMin) / (_fctGraphMax - _fctGraphMin));
			_screenToGraphScale = (decltype(_screenToGraphScale))( (_fctGraphMax - _fctGraphMin) / (_screenMax - _screenMin));
			_linConv.Update(_fctGraphMin, _fctGraphMax, _screenMin, _screenMax);
			incChangeStatus();
		}
	}
	
	void CoordinateConverter::updateGraphSize(TypeGraphCoord graphMin, TypeGraphCoord graphMax)
	{
		if (graphMin>graphMax) Swap(graphMin, graphMax);
		if ( _graphMin!=graphMin || _graphMax!=graphMax ) {
			//LOG("CoordinateConverter.update(" << graphMin << ", " << graphMax  << " )" );
			_graphMin = applyRangeLimits(graphMin);
			_graphMax = applyRangeLimits(graphMax);
			_graphRange =  tabs(_graphMax - _graphMin);
	
			_fctGraphMin = _convertFct(_graphMin);
			_fctGraphMax = _convertFct(_graphMax);
			_graphToScreenScale = (decltype(_graphToScreenScale))( (_screenMax - _screenMin) / (_fctGraphMax - _fctGraphMin));
			_screenToGraphScale = (decltype(_screenToGraphScale))( (_fctGraphMax - _fctGraphMin) / (_screenMax - _screenMin));
			
			_linConv.Update(_fctGraphMin, _fctGraphMax, _screenMin, _screenMax);
			incChangeStatus();
		}
	}
	*/
	void CoordinateConverter::Scroll( TypeScreenCoord offset ) {
		TypeGraphCoord graphMin = toGraph( _screenMin - offset );
		TypeGraphCoord graphMax = toGraph( _screenMax - offset );
		if ( isInRangeLimits(graphMin) && isInRangeLimits(graphMax)) {
			updateGraphSize( graphMin, graphMax );
		}
	}
	
	void CoordinateConverter::Zoom( double factor ) {
		TypeScreenCoord addRange=(TypeScreenCoord)((_screenRange*(factor-1.0))/2.0);
		updateGraphSize( toGraph( _screenMin - addRange ),
						 toGraph( _screenMax + addRange )
						);
	}
	
	void CoordinateConverter::SetScaleType(int t)	{
		switch(t) {
			case AXIS_SCALE_STD:
				SetConvStd();
				break;
			case AXIS_SCALE_LOG:
				SetConvLog();
				break;
			case AXIS_SCALE_POW10:
				SetConvPow10();
				break;
			default:
				SetConvStd();
				RLOG("## ERROR ## : CoordinateConverter::SetScaleType( " << t << " ) : UNKNOWN SCALE TYPE");
//				ASSERT_(false, MISSING_OR_WRONG_SCALE_TYPE);
				break;
		}
	}
	
	void CoordinateConverter::SetConvFct(TypeConvertFct convertFct, TypeConvertFct unConvertFct, const char* convTypeName) {
		_convFctName = convTypeName;
		_convertFct = convertFct;
		_unConvertFct = unConvertFct;
		ClearGraphRangeLimits();
		Update();
		incChangeStatus();
	}
	
	void CoordinateConverter::SetConvStd() {
		SetConvFct(_defautFct, _defautFct, "STD");
		_scaleType = AXIS_SCALE_STD;
		incChangeStatus();
	}
	
	void CoordinateConverter::SetConvLog()   {
		setGraphMinRangeLimit(0.00000000000001);
		Update();
		SetConvFct(_logFct,    _pow10Fct,  "LOG");
		_scaleType = AXIS_SCALE_LOG;
		updateGraphSize(getGraphMin(), getGraphMax());
		incChangeStatus();
	}
	
	void CoordinateConverter::SetConvPow10() {
		SetConvFct(_pow10Fct,  _logFct,    "10^x");
		_scaleType = AXIS_SCALE_POW10;
		incChangeStatus();
	}

}
}

