#ifndef _GraphDraw_StdElements_h_
#define _GraphDraw_StdElements_h_


#include "GraphElement.h"
#include "TickMark.h"

namespace GraphDraw_ns
{
	
	class BlankAreaElement : public GraphElement
	{
	public:
		typedef BlankAreaElement CLASSNAME;
		typedef GraphElement _B;

		BlankAreaElement() {}
		
		virtual ~BlankAreaElement() {}
		virtual void PaintElement(Draw& dw, int scale) { /* do noting */}
		virtual void SetElementWidth(int v) { _width = Upp::max(v, 3); }

	private:
		BlankAreaElement( BlankAreaElement& p)  {}
	};


	
	// ============================
	//    LabelElement   CLASS
	// ============================
#define GD_LAYDESIGN_LabelElement(CLASS, STYL) \
		template<class T>	inline CLASS& SetFont(Font v)       { STYL.labelFont = v;  return *this; }\
		template<class T>	inline CLASS& SetTextColor(Color v) { STYL.labelColor = v; return *this; }


	struct LabelElement_GEDStyle : ChStyle<LabelElement_GEDStyle>, GraphElement::GEStyle {
		Font        labelFont;
		Color       labelColor;
	};

	class LabelElement : public GraphElement
	{
	public:
		typedef LabelElement CLASSNAME;
		typedef GraphElement _B;
		
		typedef LabelElement_GEDStyle Style;

		static const Style& StyleDefault();
		static const Style& StyleTitleDefault();
		static const Style& StyleXAxisDefault();
		static const Style& StyleYAxisDefault();

		LabelElement&  SetStyle(const Style& s) {
			style = &s;
			return *this;
		}
		
	protected:
		const Style *style;
			
	public:
		Upp::String _label;

		LabelElement() {
			style = &StyleDefault();
		}
		
		LabelElement(LabelElement& p) : _B(p), style( p.style ) {}
		virtual ~LabelElement() {}

		template<class T>	inline LabelElement& SetLabel(T& v) { _label = v; return *this; }

		inline const Upp::String& GetLabel() const { return _label; }

		virtual void PaintElement(Draw& dw, int scale)
		{
			_B::PaintElementBckGround(dw, _B::GetFrame().GetSize(), style->lmntBackgnd );
			
			Font scaledFont( style->labelFont );
			scaledFont.Height(scale*scaledFont.GetHeight());
			Size sz = GetTextSize(_label, scaledFont);
			switch(_B::GetElementPos()) {
				case LEFT_OF_GRAPH:
					dw.DrawText( _B::GetElementWidth()*scale/2-sz.cy/2, _B::GetFrame().GetHeight()/2+sz.cx/2 , 900, _label, scaledFont, style->labelColor);
					break;
				case BOTTOM_OF_GRAPH:
					dw.DrawText( _B::GetFrame().GetWidth()/2-sz.cx/2, _B::GetElementWidth()*scale/2-sz.cy/2 , _label, scaledFont, style->labelColor);
					break;
				case TOP_OF_GRAPH:
					dw.DrawText( _B::GetFrame().GetWidth()/2-sz.cx/2, _B::GetElementWidth()*scale/2-sz.cy/2 , _label, scaledFont, style->labelColor);
					break;
				case RIGHT_OF_GRAPH:
					dw.DrawText( _B::GetElementWidth()*scale/2+sz.cy/2, _B::GetFrame().GetHeight()/2-sz.cx/2 , 2700, _label, scaledFont, style->labelColor);
					break;
				case FLOAT_OVER_GRAPH:
					break;
			}
		}
	};

//	// ============================
//	//    LabelElement   CLASS
//	// ============================

// QTFDisplay

//	class RichLabelElement : public GraphElement
//	{
//		private:
//		RichText    _label;
//		Color       _bckGndcolor;
//		typedef RichLabelElement                          CLASSANME;
//		typedef GraphElement _B;
//
//		public:
//		RichLabelElement() : _bckGndcolor(Null) {}
//		RichLabelElement(RichLabelElement& p) : _B(p), _bckGndcolor(p._bckGndcolor) {}
//		virtual ~RichLabelElement() {}
//
//		virtual CLASSANME* Clone() { return new CLASSANME(*this); };
//
//		template<class T>
//		inline void SetLabel(T& v) { _label = ParseQTF(v); }
////		inline const Upp::String& GetLabel() const { return _label.; }
//
//		virtual void PaintElement(Draw& dw, int scale)
//		{
//			if ( !_bckGndcolor.IsNullInstance() )
//				dw.DrawRect( 0, 0, _B::GetFrame().GetWidth(), _B::GetFrame().GetHeight(), _bckGndcolor);
//
//			//Size sz(0,0);
//			switch(_B::GetElementPos()){
//				case LEFT_OF_GRAPH:
//
//					//dw.DrawText( _B::GetElementWidth()*scale/2-sz.cy/2, _B::GetFrame().GetHeight()/2+sz.cx/2 , 900, _label, _font, _color);
//					break;
//				case BOTTOM_OF_GRAPH:
//					//dw.DrawText( _B::GetFrame().GetWidth()/2-sz.cx/2, _B::GetElementWidth()*scale/2-sz.cy/2 , _label, _font, _color);
//					break;
//				case TOP_OF_GRAPH:
//					_label.Paint(dw, 0, 0, 1000);
//					//dw.DrawText( _B::GetFrame().GetWidth()/2-sz.cx/2, _B::GetElementWidth()*scale/2-sz.cy/2 , _label, _font, _color);
//					break;
//				case RIGHT_OF_GRAPH:
//					//dw.DrawText( _B::GetElementWidth()*scale/2+sz.cy/2, _B::GetFrame().GetHeight()/2-sz.cx/2 , 2700, _label, _font, _color);
//					break;
//			}
//		}
//	};



	// ============================
	//    LegendElement   CLASS
	// ============================
#define GD_LAYDESIGN_LegendElement(CLASS, STYL) \
	inline CLASS& SetFont(Font v) { STYL.lgdFont = v;  return *this; }

	class LegendElement : public GraphElement
	{
	public:
		typedef LegendElement CLASSNAME;
		typedef GraphElement _B;

		struct Style : ChStyle<Style>, _B::GEStyle {
			Font   lgdFont;
			Color  lgdTxtColor;
			int    lgdXSeparation; // separation between two legends
			int    lgdStyleLength;
			Value  lgdFloatBackgnd;
		};
		
		static const Style& StyleDefault();
		LegendElement&  SetStyle(const Style& s) {
			style = &s;
			return *this;
		}
	protected:
		const Style *style;
		
	public:

		String _legend;

		int    _legendWeight;
		TypeVectorSeries* v_series;

		LegendElement()	: v_series(0)
		{ style = &StyleDefault(); }
		
		LegendElement(LegendElement& p)
		: _B(p)
		, style(p.style)
		, v_series(p.v_series)
		{ PrePaint(1); }

		virtual ~LegendElement() {}

		template<class T>
		inline CLASSNAME& SetLegend(T& v) { _legend = v; return *this; }


		virtual void PaintElement(Draw& dw, int scale) {
			_B::PaintElementBckGround(dw, _B::GetFrame().GetSize(), style->lmntBackgnd  );
			DrawLegend(dw, scale);
		}

		virtual void PaintFloatElement(Draw& dw, int scale){
			_B::PaintElementBckGround(dw, _B::GetFloatFrame().GetSize()*scale, style->lgdFloatBackgnd );
			DrawLegend(dw, scale);
		}

		virtual void PrePaint( int scale ) {
			if (v_series==0) {
				v_series = &_B::_parent->GetSeries();
			}

			_legendWeight = 0;
			for (int c=0; c<(*v_series).GetCount(); ++c) {
				int textLength = GetTextSize((*v_series)[c].legend, style->lgdFont).cx;
				if(_legendWeight < textLength) _legendWeight = textLength;
			}
			_legendWeight += style->lgdStyleLength + style->lgdXSeparation;
		}

		void DrawLegend(Draw& w, const int scale) const
		{
			if (v_series==0) {
				String text = "LEGEND TEXT : no series defined";
				w.DrawText( 0,0 , text, style->lgdFont, SBlack());
				return;
			}

			int nmr = fround(_B::_frame.GetSize().cx/(_legendWeight*scale));	//max number of labels per row
			if (nmr <= 0) nmr = 1;
			int nLab = (*v_series).GetCount();	//number of labels
			int idx=0;
			int ix=0;
			int iy=0;
			Font scaledFont( style->lgdFont );
			int txtHeight = scaledFont.Height(scale * style->lgdFont.GetHeight() ).GetHeight();
			while (idx<nLab) {
				while ((idx<nLab) && ((*v_series)[idx].show == false )) { ++idx; }
				
				if (idx<nLab) {
					int x = scale*ix*_legendWeight + txtHeight/2;
					int y = iy*txtHeight + txtHeight/2;
					Image img = (*v_series)[idx].MakeSerieIcon(txtHeight, scale);
					w.DrawImage(x,y, img);
					Color txtColor;
					if (style->lgdTxtColor.IsNullInstance())
						txtColor = ((*v_series)[idx].seriesPlot.IsEmpty()) ? (*v_series)[idx].markColor : (*v_series)[idx].color;
					else
						txtColor = style->lgdTxtColor;
					
					DrawText(w, x+scale*(style->lgdStyleLength+2), y, 0, (*v_series)[idx].legend, scaledFont, txtColor );
					++idx;
				}
				++ix;
				if (ix>=nmr) { ix=0; ++iy; }
			}
		}
	};

	
	// ============================
	//    MarkerElementData   CLASS
	// ============================

	class MarkerElementData : Moveable<MarkerElementData> {
	public:
		typedef Callback2< int /*markerID*/, void* > TypeMarkerUpdateCbk;


	private:
		TickMark* tickMark;
		TypeGraphCoord pos;
		int ID;
		Index<MarkerElementData*> linkedMarkers;
		void* owner;
		
		MarkerElementData(const MarkerElementData& p) : tickMark(0), pos(p.pos), ID(INVALID_MARKER_ID) {} // explicitally forbidden
		
	public:
		enum { INVALID_MARKER_ID = std::numeric_limits<decltype(ID)>::max() };

		MarkerElementData() : tickMark(0), ID(INVALID_MARKER_ID), owner(0) {}
		~MarkerElementData() {
			//GDLOGBLOCK("MarkerElementData::~MarkerElementData");
			UnlinkAll();
			if (tickMark) delete tickMark;
			tickMark = 0;
			ID = INVALID_MARKER_ID;
		}
		
		inline TickMark& GetTickMark() const { return *tickMark; }
		inline operator TickMark&()    const { return *tickMark; }

		inline TypeGraphCoord GetPos()   const { return pos; }
		inline operator TypeGraphCoord() const { return pos; }
		
		MarkerElementData& operator=( const TypeGraphCoord v ) { pos = v; return *this; }
		MarkerElementData& operator=( const MarkerElementData v ) { pos = v.pos; return *this; }
		MarkerElementData& operator=( const MarkerElementData* v ) { pos = v->pos; return *this; }

		inline int GetID() const { return ID; }
		
		template <class T>
		T& CreateMarker(int id, void* own) // Only valid ID values allowed
		{
			owner = own;
			ID = id;
			if (tickMark) delete tickMark;
			T& tick = * new T();
			tickMark = &tick;
			return tick;
		}
		
		inline bool IsLinkedTo(MarkerElementData* m) const { return ((linkedMarkers.Find(m)  >= 0) ? true : false ); }
		inline bool IsLinkedTo(MarkerElementData& m) const { return ((linkedMarkers.Find(&m) >= 0) ? true : false ); }
		
		void Link(MarkerElementData* m) {
			//GDLOGBLOCK("MarkerElementData::Link");
			if (m == 0)  return;
			if (linkedMarkers.Find(m) >= 0) return;
			linkedMarkers.Add( m );
			m->Link(this);
		}
		inline void Link(MarkerElementData& m) { Link(&m); }
	
		void Unlink(MarkerElementData* m) {
			//GDLOGBLOCK("MarkerElementData::Unlink  REQUESTED");
			if (m == 0)  return;
			if ((linkedMarkers.Find(m)) < 0) return;
			//LOG("MarkerElementData::Unlink  removing link  count=" << linkedMarkers.GetCount());
			linkedMarkers.RemoveKey( m );
			//LOG("MarkerElementData::Unlink  removed        count=" << linkedMarkers.GetCount());
			m->Unlink(this);
		}
		inline void Unlink(MarkerElementData& m) { Unlink(&m); }
	
		void UnlinkAll() {
			//GDLOGBLOCK("MarkerElementData::UnlinkAll");
			while( linkedMarkers.GetCount() > 0) {
				Unlink(linkedMarkers[0]);
			}
		}
		
		void UpdateLinked(TypeMarkerUpdateCbk& cbk) {
			//GDLOGBLOCK("MarkerElementData::UnlinkAll");
			for (int c=(linkedMarkers.GetCount()-1); c>=0; --c) {
				linkedMarkers[c]->pos = pos;
				cbk( linkedMarkers[c]->ID, owner );
			}
		}
	};

	// ============================
	//    MarkerPosList
	// ============================
	typedef ArrayMap<int, MarkerElementData> MarkerPosList;

	// ============================
	//    MarkerElement   CLASS
	// ============================
	class MarkerElement : public GraphElement
	{
	public:
		typedef GraphElement  _B;
		typedef MarkerElement  CLASSNAME;

		struct Style : ChStyle<Style>, _B::GEStyle {
			Color       mrkLineColor;
			Color       mrkTickColor;
			int         mrkLineWidth;
			//Function<TickMark* ()> markerFactory;
		};
		
		static const Style& StyleDefault();
		MarkerElement&  SetStyle(const Style& s) {
			style = &s;
			return *this;
		}

	protected:
		const Style *style;
		
	public:
		CoordinateConverter& _coordConverter;
		MarkerPosList markers;

		MarkerElement(CoordinateConverter& coordconv)
		: _coordConverter(coordconv)
		{
			style = &StyleDefault();
			_B::DisablePos(FLOAT_OVER_GRAPH);
		}
		
		MarkerElement(MarkerElement& p)
		: _B(p)
		, style(p.style)
		, _coordConverter(p._coordConverter)
		{
			_B::DisablePos(FLOAT_OVER_GRAPH);
		}
		
		virtual ~MarkerElement() {}

		template<class T>	GD_DEPRECATED("A remplacer SetMarkColor, SetLineColor")   inline MarkerElement& SetTextColor(T v)  { return *this; }

		template <class MARKER_TYPE>
		MARKER_TYPE&  AddMarker(int markerId, const String txt, TypeGraphCoord pos=0.0) {
			MarkerElementData& markData = markers.Add(markerId);
			markData = pos;
			MARKER_TYPE& m = markData.CreateMarker<MARKER_TYPE>(markerId, this);
			m.SetText(txt);
			return m;
		}

		template <class MARKER_TYPE>
		MARKER_TYPE&  AddMarker(int markerId, TypeGraphCoord pos=0.0) {
			MarkerElementData& markData = markers.Add(markerId);
			markData = pos;
			return markData.CreateMarker<MARKER_TYPE>(markerId, this);
		}


		SmartTextTickMark&  AddMarker(int markerId, TypeGraphCoord pos=0.0) {
			MarkerElementData& markData = markers.Add(markerId);
			markData = pos;
			return markData.CreateMarker<SmartTextTickMark>(markerId, this);
		}

		SmartTextTickMark&  AddMarker(int markerId, const String txt, TypeGraphCoord pos=1.0) {
			MarkerElementData& markData = markers.Add(markerId);
			markData = pos;
			SmartTextTickMark& m = markData.CreateMarker<SmartTextTickMark>(markerId, this);
			m.SetText(txt);
			return m;
		}

		void ClearMarkers() {
			markers.Clear();
		}

		virtual void  SetElementPos(ElementPosition v) {
			if (v==LEFT_OF_GRAPH || v==RIGHT_OF_GRAPH) {
				_B::DisablePos(BOTTOM_OF_GRAPH);
				_B::DisablePos(TOP_OF_GRAPH);
			} else {
				_B::DisablePos(LEFT_OF_GRAPH);
				_B::DisablePos(RIGHT_OF_GRAPH);
			}
			_B::SetElementPos(v);
		}


		virtual void PaintHLine(Draw& dw, int range)
		{
			if ( ! (style->mrkLineColor.IsNullInstance()) )
			{
				MarkerPosList::Iterator iter = markers.Begin();
				MarkerPosList::ConstIterator endIter =  markers.End();
				while( iter != endIter)
				{
					MarkerElementData& markerData = *iter;
					if (_coordConverter.IsInGraphVisibleRange(markerData)) {
						const TypeScreenCoord y = _coordConverter.toScreen(markerData);
						dw.DrawLineOp( 0, y, range, y, style->mrkLineWidth, style->mrkLineColor );
					}
					++iter;
				}
			}
		}

		virtual void PaintVLine(Draw& dw, int range)
		{
			if (! (style->mrkLineColor.IsNullInstance()) )
			{
				MarkerPosList::Iterator iter = markers.Begin();
				MarkerPosList::ConstIterator endIter =  markers.End();
				while( iter != endIter)
				{
					MarkerElementData& markerData = *iter;
					if (_coordConverter.IsInGraphVisibleRange(markerData)) {
						const TypeScreenCoord x = _coordConverter.toScreen(markerData);
						dw.DrawLineOp( x, 0, x, range, style->mrkLineWidth, style->mrkLineColor );
					}
					++iter;
				}
			}
		}

		virtual void PaintElement(Draw& dw, int scale)
		{
			_B::PaintElementBckGround(dw, _B::GetFrame().GetSize(), style->lmntBackgnd );

			//dw.DrawLineOp(_B::GetElementWidth()*scale, _coordConverter.getScreenMin(), _B::GetElementWidth()*scale, _coordConverter.getScreenMax(), tyle->mrkLineWidth*scale, _color );
			MarkerPosList::Iterator iter = markers.Begin();
			MarkerPosList::ConstIterator endIter = markers.End();
			int c=0;
			while ( iter != endIter ) {
					MarkerElementData& markerData = *iter;
				if (_coordConverter.IsInGraphVisibleRange(markerData)) {
					switch( _B::GetElementPos() ) {
						case LEFT_OF_GRAPH:
							markerData.GetTickMark().Paint(dw, _B::GetElementPos(), scale, _B::GetElementWidth()*scale, _coordConverter.toScreen(markerData), style->mrkTickColor);
							break;
						case BOTTOM_OF_GRAPH:
							markerData.GetTickMark().Paint(dw, _B::GetElementPos(), scale, _coordConverter.toScreen(markerData), 0, style->mrkTickColor );
							break;
						case TOP_OF_GRAPH:
							markerData.GetTickMark().Paint(dw, _B::GetElementPos(), scale,  _coordConverter.toScreen(markerData), _B::GetElementWidth()*scale, style->mrkTickColor );
							break;
						case RIGHT_OF_GRAPH:
							markerData.GetTickMark().Paint(dw, _B::GetElementPos(), scale, 0, _coordConverter.toScreen(markerData), style->mrkTickColor );
							break;
						case FLOAT_OVER_GRAPH:
							break;
					}
				}
				++iter;
				++c;
			}
		}
		
		virtual void PaintOnPlot_overData(Draw& dw, int otherWidth, int scale)
		{
			switch( _B::GetElementPos() )
			{
				case LEFT_OF_GRAPH:
				case RIGHT_OF_GRAPH:
					PaintHLine(dw, otherWidth);
					break;
				case TOP_OF_GRAPH:
				case BOTTOM_OF_GRAPH:
					PaintVLine(dw, otherWidth);
					break;
				case FLOAT_OVER_GRAPH:
					break;
			}
			
		}
	};
}
#endif
