#include "VirtualTerminal.h"

#include <errno.h>
#include <fcntl.h>
#include <linux/kd.h>
#include <linux/vt.h>
#include <sys/ioctl.h>
#include <unistd.h>

#define LLOG(x)	// RLOG("VirtualTerminal: " << x)

using namespace Upp;

#ifndef KDSKBMUTE
# define KDSKBMUTE 0x4B51
#endif

static std::atomic_int  vt_handle;
static std::atomic_bool has_focus;

static void sGotFocus(int sig)
{
	if (vt_handle < 0 || ioctl(vt_handle, VT_RELDISP, VT_ACKACQ) < 0)
		return;
	has_focus = true;
}

static void sLostFocus(int sig)
{
	if (vt_handle < 0 || ioctl(vt_handle, VT_RELDISP, 1) < 0)
		return;
	has_focus = false;
}

bool VirtualTerminal::HasFocus() const
{
	return has_focus;
}

bool VirtualTerminal::SetFocus(int vt_number)
{
	if (vt_handle < 0)
		return false;

	do {
		errno = 0;
		if (ioctl(vt_handle, VT_ACTIVATE, vt_number) < 0 && errno != EINTR) {
			LLOG("ioctl(VT_ACTIVATE) failed: " << GetLastErrorMessage());
			return false;
		}

		if (ioctl(vt_handle, VT_WAITACTIVE, vt_number) < 0 && errno != EINTR) {
			LLOG("ioctl(VT_WAITACTIVE) failed: " << GetLastErrorMessage());
			return false;
		}
	} while(errno == EINTR);

	LLOG("Switched to /dev/tty" << vt_number);
	return true;
}

int VirtualTerminal::Allocate()
{
	int fd = -1;
	int vt_number = -1;
	struct vt_stat vtstat;

	if ((fd = open("/dev/tty0", O_RDWR)) < 0) {
		LLOG("Couldn't open /dev/tty0: " << GetLastErrorMessage());
		return -1;
	}

	if (ioctl(fd, VT_GETSTATE, &vtstat) < 0) {
		LLOG("ioctl(VT_GETSTATE) failed: " << GetLastErrorMessage());
		close(fd);
		return -1;
	}

	LLOG("Current vt is /dev/tty" << vtstat.v_active);

	previous_vt = vtstat.v_active;

	// Since we can directly use only F1-F12 keys, let's
	// clamp the valid tty number within tty1-tty12 range.

	const int vt_min = MIN_NR_CONSOLES;
	const int vt_max = min(12, MAX_NR_CONSOLES);
	
	if (ioctl(fd, VT_OPENQRY, &vt_number) < 0 || vt_min > vt_number || vt_number > vt_max) {
		LLOG("Failed to find a free tty in rainge tty1-tty12. " << GetLastErrorMessage());
		close(fd);
		return -1;
	}

	LLOG("Found free vt: /dev/tty" << vt_number);

	close(fd);
	return vt_number;
}

void VirtualTerminal::Deallocate()
{
	has_focus = false;

	if (active_vt > 0)
		if (ioctl(vt_handle, VT_DISALLOCATE, active_vt) < 0) {
			LLOG("Failed to disallocate /dev/tty" << active_vt << ": " << GetLastErrorMessage());
		}

	fflush(stdout);
	close(vt_handle);

	vt_handle   = -1;
	active_vt   = -1;
	previous_vt = -1;
}

bool VirtualTerminal::SetMode()
{
	struct vt_mode vtmode;
	Zero(vtmode);

	if (ioctl(vt_handle, VT_GETMODE, &vtmode) < 0) {
		LLOG("ioctl(VT_GETMODE) failed: " << GetLastErrorMessage());
		return false;
	}

	struct sigaction sigact1, sigact2;
	Zero(sigact1);
	Zero(sigact2);

	sigact1.sa_handler = sGotFocus;
	sigact1.sa_flags   = SA_RESTART;
	sigaction(SIGUSR1, &sigact1, nullptr);

	sigact2.sa_handler = sLostFocus;
	sigact2.sa_flags   = SA_RESTART;
	sigaction(SIGUSR2, &sigact2, nullptr);

	vtmode.mode   = VT_PROCESS;
	vtmode.acqsig = SIGUSR1;
	vtmode.relsig = SIGUSR2;

	if (ioctl(vt_handle, VT_SETMODE, &vtmode) < 0) {
		LLOG("ioctl(VT_SETMODE: VT_PROCESS) failed: " << GetLastErrorMessage());
		return false;
	}

	return true;
}

void VirtualTerminal::ResetMode()
{
	struct vt_mode vtmode;
	Zero(vtmode);

	vtmode.mode = VT_AUTO;

	struct sigaction sigact1, sigact2;
	Zero(sigact1);
	Zero(sigact2);

	sigact1.sa_handler = SIG_DFL;
	sigaction(SIGUSR1, &sigact1, nullptr);

	sigact2.sa_handler = SIG_DFL;
	sigaction(SIGUSR2, &sigact2, nullptr);

	if (ioctl(vt_handle, VT_SETMODE, &vtmode) < 0) {
		LLOG("ioctl(VT_SETMODE: VT_AUTO) failed: " << GetLastErrorMessage());
	}
}

bool VirtualTerminal::Open()
{
	Close();

	int vt_number = Allocate();

	if (vt_number < 0)
		return false;

	if ((vt_handle = open(Format("/dev/tty%d", vt_number), O_RDWR)) <  0) {
		LLOG("Failed to open /dev/tty" << vt_number << ": " << GetLastErrorMessage());
		return false;
	}

	if(!SetFocus(vt_number))
		return false;

	active_vt = vt_number;
	has_focus = true;

	if (!SetMode())
		return false;

	if (ioctl(vt_handle, KDGETMODE, &previous_mode) < 0){
		LLOG("Failed to get current VT mode: " << GetLastErrorMessage());
	}

	if (previous_mode != KD_GRAPHICS && ioctl(vt_handle, KDSETMODE, KD_GRAPHICS) < 0){
		LLOG("Failed to set VT graphics mode: " << GetLastErrorMessage());
	}

	if (tcgetattr(vt_handle, &terminal_attributes) < 0){
		LLOG("Failed to read terminal attributes: " << GetLastErrorMessage());
	}

	struct termios attributes = terminal_attributes;
	cfmakeraw(&attributes);
	attributes.c_oflag |=  OPOST;

	if (tcsetattr(vt_handle, TCSANOW, &attributes) < 0){
		LLOG("Failed to set terminal attributes: " << GetLastErrorMessage());
	}

	if (ioctl(vt_handle, KDGKBMODE, &previous_keyboard_mode) < 0) {
		LLOG("Failed to get keyboard mode: " << GetLastErrorMessage());
	}

	if (ioctl(vt_handle, KDSKBMUTE, 1) < 0) {
		if (ioctl(vt_handle, KDSKBMODE, K_OFF) < 0)
			if (ioctl(vt_handle, KDSKBMODE, K_RAW) < 0)
				LLOG("Failed to set keyboard mode: " << GetLastErrorMessage());
	}

	LLOG("/dev/tty" << active_vt << "is successfully opened.");

	fflush(stdout);
	return true;
}

void VirtualTerminal::Close()
{
	if (vt_handle < 0) return;

	if (ioctl(vt_handle, KDSETMODE, &previous_mode) < 0){
		LLOG("Failed to restore VT mode: " << GetLastErrorMessage());
	}

	if (ioctl(vt_handle, KDSKBMUTE, 0) < 0)
		ioctl(vt_handle, KDSKBMODE, K_XLATE);

	if (ioctl(vt_handle, KDSKBMODE, previous_keyboard_mode) < 0){
		LLOG("Failed to restore keyboard mode: " << GetLastErrorMessage());
	}

	if (tcsetattr(vt_handle, TCSANOW, &terminal_attributes) < 0){
		LLOG("Failed to restore terminal attributes: " << GetLastErrorMessage());
	}

	ResetMode();
	SetFocus(previous_vt);
	Deallocate();
}

VirtualTerminal::VirtualTerminal()
{
	vt_handle = -1;
	has_focus = false;
}
