#ifndef _GraphCtrl_CoordinateConverter_h_
#define _GraphCtrl_CoordinateConverter_h_



namespace GraphDraw_ns
{

	class CoordinateConverterOwner {
		public:
		virtual void RefreshWhenLinkUpdate() = 0;
		
	};


	class ChangeStatus {
		private:
			unsigned int status;
			ChangeStatus(const ChangeStatus& p) : status(0) {}

		public:
			ChangeStatus() : status(0) {}
			
			ChangeStatus& incChangeStatus() {
				++status;
				return *this;
			}
			
			inline bool hasChangedUpdate(const ChangeStatus& p) {
				if ( status == 0 || status != p.status )  {
					status = p.status;
					return true;
				}
				return false;
			}

			inline bool hasChanged(const ChangeStatus& p) {
				if ( status != p.status )  return true;
				return false;
			}
	};
	
	typedef enum {
		LINK_UPDATE_RANGE  = 0x01,
		LINK_UPDATE_LIMITS = 0x02,
		LINK_UPDATE_RANGE_LIMITS = LINK_UPDATE_RANGE + LINK_UPDATE_LIMITS,
	} LinkUpdateStrategy;
	
	class  LinearConverter {
		private:
			TypeGraphCoord toGraphScale;
			TypeGraphCoord toScreenScale;
			TypeGraphCoord toGraphOffset;
			TypeGraphCoord toScreenOffset;
			bool isInverted;

		public:
			LinearConverter() : toGraphScale(1), toScreenScale(1), toGraphOffset(0), toScreenOffset(0), isInverted(false) {}
			LinearConverter(const LinearConverter& p) : toGraphScale(p.toGraphScale), toScreenScale(p.toScreenScale), toGraphOffset(p.toGraphOffset), toScreenOffset(p.toScreenOffset), isInverted(p.isInverted) {}
			
			void SetInverted(bool v=true) { isInverted=v; }
			
			void Update(TypeGraphCoord gLeft, TypeGraphCoord gRight, const TypeScreenCoord sMin, const TypeScreenCoord sMax) {
				if (isInverted) Swap(gLeft, gRight);
				toScreenScale = (TypeGraphCoord)( (sMax - sMin) / (gRight - gLeft));
				toGraphScale = (TypeGraphCoord)( (gRight - gLeft) / (sMax - sMin));
				
				toScreenOffset = sMin - toScreenScale*gLeft;
				toGraphOffset  = gLeft - toGraphScale*sMin;
			}

			inline TypeScreenCoord toScreen(const TypeGraphCoord v) const { return (TypeScreenCoord)(v*toScreenScale + toScreenOffset); }
			inline TypeGraphCoord  toGraph(const TypeScreenCoord v) const { return (TypeGraphCoord)(v*toGraphScale  + toGraphOffset); }
			inline void toGraph(const TypeScreenCoord* v, TypeGraphCoord* out, const int n) const {
				const TypeScreenCoord* const iterEnd = --v+n;
				--out;
				while (v<iterEnd) {
					*(++out) = toGraph(*(++v));
				}
			}
	};

	// ============================
	//    CoordinateConverter   CLASS
	// ============================
	class CoordinateConverter : public ChangeStatus
	{
		public:
			typedef enum {
				AXIS_SCALE_STD = 0,
				AXIS_SCALE_LOG,
				AXIS_SCALE_POW10,
			} AxisScaleType;

		private:
			// copy constructor forbidden
			CoordinateConverter& operator=(const CoordinateConverter& c) {	return *this;}
			CoordinateConverter(const CoordinateConverter& p) : _owner(p._owner), _linConv(p._linConv) {}

		protected:
			
			CoordinateConverterOwner& _owner;
			bool _isZoomAllowed;
			bool _isScrollAllowed;
			TypeScreenCoord _screenMin;
			TypeScreenCoord _screenMax;
			TypeScreenCoord _screenRange;
			TypeGraphCoord _graphToScreenScale;
			TypeGraphCoord _graphMin;
			TypeGraphCoord _graphMax;
			TypeGraphCoord _graphRange;
			TypeGraphCoord _screenToGraphScale;
			Value _graphMaxLimit;
			Value _graphMinLimit;
			LinearConverter _linConv;
			
			TypeGraphCoord _fctGraphMin;
			TypeGraphCoord _fctGraphMax;
			typedef TypeGraphCoord (*TypeConvertFct) (TypeGraphCoord);
			TypeConvertFct _convertFct;
			TypeConvertFct _unConvertFct;
			const char* _convFctName;
			int _scaleType;

			static TypeGraphCoord _defautFct(TypeGraphCoord v) { return v; }
			static TypeGraphCoord _logFct(TypeGraphCoord v)    { if (v<=0) v=0.00001; return log10(v); }
			static TypeGraphCoord _pow10Fct(TypeGraphCoord v)  { return pow(10.0,v); }

		public:
			typedef CoordinateConverter CLASSNAME;
			CoordinateConverter(CoordinateConverterOwner& owner);
			CoordinateConverter(CoordinateConverterOwner& owner, TypeGraphCoord graphMin, TypeGraphCoord graphMax, TypeScreenCoord screenMin, TypeScreenCoord screenMax);
			virtual ~CoordinateConverter();

			Index<CoordinateConverter*> links;
			UpdateCounter<CoordinateConverter> linkStatusCounter;
			
			void UpdateLinks(LinkUpdateStrategy strategy);

			static void Link(CoordinateConverter& local, CoordinateConverter& other) {
				if (local.Link(other) ) {
					other.Link(local);
				}
			}
			
			static void Unlink(CoordinateConverter& local, CoordinateConverter& other) {
				local.Unlink(other);
			}
			
			bool Link(CoordinateConverter& conv);
			void Unlink(CoordinateConverter& conv);
			void UnlinkAll();
			
			inline TypeScreenCoord getScreenMin()        const { return _screenMin; }
			inline TypeScreenCoord getScreenMax()        const { return _screenMax; }
			inline TypeScreenCoord getScreenRange()      const { return _screenRange; }
			inline TypeGraphCoord  getGraphMin()         const { return _graphMin; }
			inline TypeGraphCoord  getGraphMax()         const { return _graphMax; }
			inline Value           getGraphMinRangeLimit()    const { return _graphMinLimit; }
			inline Value           getGraphMaxRangeLimit()    const { return _graphMaxLimit; }
			inline TypeGraphCoord  getGraphRange()       const { return _graphRange; }
			inline TypeGraphCoord  getSignedGraphRange() const { return (_graphMax-_graphMin); }
			inline bool IsInGraphVisibleRange(TypeGraphCoord p) const { return ((_graphMin <= p) && (p <= _graphMax)); }

			inline void AllowZoom(bool p = true)               { _isZoomAllowed = p; }
			inline void AllowScroll(bool p = true)             { _isScrollAllowed = p; }
			inline bool IsZoomAllowed()                  const { return _isZoomAllowed; }
			inline bool IsScrollAllowed()                const { return _isScrollAllowed; }


			Callback MakeRestoreAxisMinMaxCB() {
				return THISBACK2(updateGraphSize, _graphMin, _graphMax);
			}
			inline void setGraphMaxRangeLimit(TypeGraphCoord v) { _graphMaxLimit = v; }
			inline void setGraphMinRangeLimit(TypeGraphCoord v) { _graphMinLimit = v; }
			inline void ClearGraphRangeLimits() { _graphMaxLimit = Null; _graphMinLimit = Null; }

			virtual TypeGraphCoord applyRangeLimits(TypeGraphCoord v);
			virtual bool isInRangeLimits(TypeGraphCoord v);

			
			inline void Update() {
				update( getGraphMin(), getGraphMax(), getScreenMin(), getScreenMax() );
			}

			inline void SetGraphMin(TypeGraphCoord v) { update( v, _graphMax, _screenMin, _screenMax, false ); }
			inline void SetGraphMax(TypeGraphCoord v) { update( _graphMin, v, _screenMin, _screenMax, false ); }
			virtual void update(TypeGraphCoord graphMin, TypeGraphCoord graphMax, TypeScreenCoord screenMin, TypeScreenCoord screenMax, bool doSwap=true);
			inline void updateScreenSize(TypeScreenCoord screenMin, TypeScreenCoord screenMax) { update( _graphMin, _graphMax, screenMin, screenMax ); }
			inline void updateGraphSize(TypeGraphCoord graphMin, TypeGraphCoord graphMax)      { update( graphMin, graphMax, _screenMin, _screenMax ); }
			virtual void Scroll( TypeScreenCoord offset );
			virtual void Zoom( double factor );

			void SetInverted(bool v=true) { _linConv.SetInverted(v); Update(); }
			virtual void SetScaleType(int t);
			inline int GetScaleType() const { return _scaleType; }
			String GetConversionType() const { return _convFctName; }

			void SetConvFct(TypeConvertFct convertFct, TypeConvertFct unConvertFct, const char* convTypeName);
			void SetConvStd();
			void SetConvLog();
			void SetConvPow10();

			inline TypeScreenCoord toScreen(const TypeGraphCoord v) const { return _linConv.toScreen( _convertFct(v) ); }
			inline TypeGraphCoord  toGraph(const TypeScreenCoord v) const { return _unConvertFct( _linConv.toGraph(v) ); }
	};

	template<class STREAM>
	STREAM& operator<<(STREAM& str, CoordinateConverter v)
	{
		str << "\n CoordinateConverter :   " << v.GetConversionType();
		str << "\n     - G[" << v.getGraphMin() << ", " << v.getGraphMax() << "] ==> range = " << v.getGraphRange();
		str << "\n     - S[" << v.getScreenMin() << ", " << v.getScreenMax()<< "]  ==> range = " << v.getScreenRange();

		str << "\n     - TEST :  G["<< v.getGraphMin() <<                          "] => S " << v.toScreen(v.getGraphMin());
		str << "\n               G["<< (v.getGraphMin()+v.getGraphMax())/2.0 <<    "] => S " << v.toScreen((v.getGraphMin()+v.getGraphMax())/2.0);
		str << "\n               G["<< v.getGraphMax() <<                          "] => S " << v.toScreen(v.getGraphMax());
		str << "\n               S["<< v.getScreenMin() <<                         "] => G " << v.toGraph(v.getScreenMin());
		str << "\n               S["<< (v.getScreenMin()+v.getScreenMax())/2.0 <<  "] => G " << v.toGraph((v.getScreenMin()+v.getScreenMax())/2.0);
		str << "\n               S["<< v.getScreenMax() <<                         "] => G " << v.toGraph(v.getScreenMax());

		return str;
	}
/*
	// ============================
	//    CoordinateConverter   CLASS
	// ============================
	class LogCoordinateConverter : public CoordinateConverter
	{
		private:
			typedef CoordinateConverter _B;
			TypeGraphCoord _logGraphMin;
			TypeGraphCoord _logGraphMax;

		public:
			LogCoordinateConverter(CoordinateConverterOwner& owner)
			: CoordinateConverter(owner)
			{
				setGraphMinRangeLimit(0.0000000001);
				update( 0.1, 100, 0, 100);
			}

			LogCoordinateConverter(CoordinateConverterOwner& owner, TypeGraphCoord graphMin, TypeGraphCoord graphMax, TypeScreenCoord screenMin, TypeScreenCoord screenMax)
			: CoordinateConverter(owner)
			{
				setGraphMinRangeLimit(0.0000000001);
				update( graphMin, graphMax, screenMin, screenMax);
			}

			virtual void update(TypeGraphCoord graphMin, TypeGraphCoord graphMax, TypeScreenCoord screenMin, TypeScreenCoord screenMax)
			{
				_B::update(graphMin, graphMax, screenMin, screenMax);

				_logGraphMin = log10(_graphMin);
				_logGraphMax = log10(_graphMax);
				_graphToScreenScale = (screenMax - screenMin) / (_logGraphMax - _logGraphMin);
				_screenToGraphScale = (_logGraphMax - _logGraphMin) / (screenMax - screenMin);
				incChangeStatus();
			}

			virtual void updateScreenSize(TypeScreenCoord screenMin, TypeScreenCoord screenMax)
			{
				_B::updateScreenSize(screenMin, screenMax);

				_graphToScreenScale = (screenMax - screenMin) / (_logGraphMax - _logGraphMin);
				_screenToGraphScale = (_logGraphMax - _logGraphMin) / (screenMax - screenMin);
				incChangeStatus();
			}


			virtual void updateGraphSize(TypeGraphCoord graphMin, TypeGraphCoord graphMax)
			{
				_B::updateGraphSize(graphMin, graphMax);

				_logGraphMin = log10(_graphMin);
				_logGraphMax = log10(_graphMax);
				_graphToScreenScale = (_B::_screenMax - _B::_screenMin) / (_logGraphMax - _logGraphMin);
				_screenToGraphScale = (_logGraphMax - _logGraphMin) / (_B::_screenMax - _B::_screenMin);
				incChangeStatus();
			}

			virtual TypeScreenCoord toScreen(TypeGraphCoord v) const
			{
				if (v<=0) return _B::_screenMin;
				return  ((TypeScreenCoord)((log10(v)-_logGraphMin)*_B::_graphToScreenScale + _B::_screenMin));
			}

			virtual TypeGraphCoord toGraph(TypeScreenCoord v) const { return  (TypeGraphCoord)(pow(10, (v-_B::_screenMin)*_B::_screenToGraphScale + _logGraphMin)); }
			virtual const char* GetConversionType() const { return "LOG"; }
	};

	class Pow10CoordinateConverter : public CoordinateConverter
	{
		private:
			typedef CoordinateConverter _B;
			long double _pow10GraphMin;
			long double _pow10GraphMax;

		public:
			Pow10CoordinateConverter(CoordinateConverterOwner& owner) 
			: CoordinateConverter(owner)
			{
				update( 0, 100, 0, 100);
			}

			Pow10CoordinateConverter(CoordinateConverterOwner& owner, TypeGraphCoord graphMin, TypeGraphCoord graphMax, TypeScreenCoord screenMin, TypeScreenCoord screenMax)
			: CoordinateConverter(owner)
			{
				update( graphMin, graphMax, screenMin, screenMax);
			}

			virtual void update(TypeGraphCoord graphMin, TypeGraphCoord graphMax, TypeScreenCoord screenMin, TypeScreenCoord screenMax) {
				_B::update(graphMin, graphMax, screenMin, screenMax);
				_pow10GraphMin = pow(10, _graphMin);
				_pow10GraphMax = pow(10, _graphMax);
				_graphToScreenScale = (decltype(_B::_graphToScreenScale))( (screenMax - screenMin) / (_pow10GraphMax - _pow10GraphMin));
				_screenToGraphScale = (decltype(_B::_screenToGraphScale))( (_pow10GraphMax - _pow10GraphMin) / (screenMax - screenMin));
				incChangeStatus();
			}

			virtual TypeScreenCoord toScreen(TypeGraphCoord v) const { return  ((TypeScreenCoord)((pow(10, v)-_pow10GraphMin)*_B::_graphToScreenScale + _B::_screenMin)); }
			virtual TypeGraphCoord toGraph(TypeScreenCoord v)  const { return  (TypeGraphCoord)(log10((v-_B::_screenMin)*_B::_screenToGraphScale + _pow10GraphMin)); }
			virtual const char* GetConversionType() const      { return "10^x"; }
	};
	*/
} // namespace GraphDraw_ns

typedef GraphDraw_ns::CoordinateConverter GraphGConv; // Needed by '.USC' file

#endif
