#include <CtrlLib/CtrlLib.h>
#include <Markdown/Markdown.h>

using namespace Upp;

class MarkdownEditor : public TopWindow {
	Splitter     splitter;
	LineEdit     editor;
	RichTextView view;
	MenuBar      mainmenu;
	StatusBar	 statusbar;
	String		 filename;

public:
	MarkdownEditor()
	{
		Title(t_("Simple Markdown Editor"));
		Sizeable().Zoomable().CenterScreen().SetRect(0, 0, 1024, 768);
		AddFrame(mainmenu);
		AddFrame(statusbar);
		mainmenu.Set([=](Bar& menu) { MainMenu(menu); });
		Add(splitter.SizePos());
		splitter.Set(editor, view);
		editor.SetFont(MonospaceZ(13));
		view.SetZoom(GetRichTextStdScreenZoom());
	}
	
	void MainMenu(Bar& menu)
	{
		menu.Sub(t_("File"), [=](Bar& menu)
		{
			menu.Add(t_("Open"), [=] { OpenFile(); }).Key(K_CTRL_O);
			menu.Add(editor.IsDirty(),t_("Save"), [=] { Save(); }).Key(K_CTRL_S);
			menu.Add(t_("Save as.."), [=] { SaveAs(); }).Key(K_ALT_S);
			menu.Separator();
			menu.Add(t_("Export as Qtf.."), [=] { Export(); }).Key(K_CTRL_X);
			menu.Separator();
			menu.Add(t_("Exit"), [=] { Close(); }).Key(K_CTRL_Q);
		});
	}
	
	void OpenFile()
	{
		if(editor.IsDirty())
			if(PromptYesNo(t_("Do you really want to discard the changes and open a new markdown file?")) == IDNO)
				return;
		String fname = SelectFileOpen("*.md");
		if(!fname.IsVoid()) {
			filename = fname;
			editor.Set(LoadFile(filename));
			Title(filename);
			ConvertMarkdownToQtf();
		}
		else
			statusbar.Temporary(t_("File open aborted."));
	}
	
	void Save0()
	{
		if(SaveFile(filename, ~editor)) {
			editor.ClearModify();
			Title(filename);
			statusbar.Temporary(t_("File saved to disk..."));
		}
		else ErrorOK(Format(t_("Couldn't save file [* %s]"), DeQtf(filename)));

	}
	
	void Save()
	{
		filename.IsEmpty() ? SaveAs() : Save0();
	}

	void SaveAs()
	{
		String fname = SelectFileSaveAs("*.md");
		if(!fname.IsEmpty()) {
			filename = fname;
			Save0();
		}
		else
			statusbar.Temporary(t_("File save aborted."));
	}
	
	void Export()
	{
		if(!SelectSaveFile("*.qtf", ~view))
			statusbar.Temporary(t_("File export aborted."));
	}

	void ConvertMarkdownToQtf()
	{
		view.SetQTF(MarkdownConverter().Tables().ToQtf(~editor));
	}
	
	bool Key(dword key, int) override
	{
		ConvertMarkdownToQtf();
		return false;
	}
};

GUI_APP_MAIN
{
	MarkdownEditor().Run();
}
