/**
* IdleTracker - a DLL that tracks the user's idle input time
*               system-wide.
*
* Usage
* =====
* - call IdleTrackerInit() when you want to start monitoring.
* - call IdleTrackerTerm() when you want to stop monitoring.
* - to get the time past since last user input, do the following:
*    GetTickCount() - IdleTrackerGetLastTickCount()
*
* Author: Sidney Chong
* Date: 25/5/2000
* Version: 1.0
**/

#include "IdleTracker.h"

/**
* The following global data is SHARED among all instances of the DLL
* (processes); i.e., these are system-wide globals.
**/ 
#pragma data_seg(".IdleTracer")	// you must define as SHARED in .def
HHOOK 	g_hHkKeyboard = NULL;	// handle to the keyboard hook
HHOOK 	g_hHkMouse = NULL;	// handle to the mouse hook
DWORD	g_dwLastTick = 0;	// tick time of last input event
LONG	g_mouseLocX = -1;	// x-location of mouse position
LONG	g_mouseLocY = -1;	// y-location of mouse position
int    g_procCount = 0;
#pragma data_seg()
#pragma comment(linker, "/section:.IdleTracer,rws")

HINSTANCE g_hInstance = NULL;   // global instance handle

/**
* Get tick count of last keyboard or mouse event
**/
DLLIMPORT double IdleTrackerIdleTime()
{
	DWORD cur = GetTickCount();
	if (cur > g_dwLastTick)
		return (double)(cur - g_dwLastTick) / 1000.0;
	else
		return (double)(cur + (0xffffffff - g_dwLastTick) + 1) / 1000.0;
}

DLLIMPORT void IdleTrackerReset()
{
		g_dwLastTick = GetTickCount();
}

/**
* Keyboard hook: record tick count
**/
LRESULT CALLBACK KeyboardTracker(int code, WPARAM wParam, LPARAM lParam)
{
	if (code==HC_ACTION) {
		g_dwLastTick = GetTickCount();
	}
	return ::CallNextHookEx(g_hHkKeyboard, code, wParam, lParam);
}

/**
* Mouse hook: record tick count
**/
LRESULT CALLBACK MouseTracker(int code, WPARAM wParam, LPARAM lParam)
{
	if (code==HC_ACTION) {
		MOUSEHOOKSTRUCT* pStruct = (MOUSEHOOKSTRUCT*)lParam;
		//we will assume that any mouse msg with the same locations as spurious
		if (abs(pStruct->pt.x - g_mouseLocX) > 10 || abs(pStruct->pt.y - g_mouseLocY) > 10)
		{
			g_dwLastTick = GetTickCount();
		}
		g_mouseLocX = pStruct->pt.x;
		g_mouseLocY = pStruct->pt.y;
	}
	return ::CallNextHookEx(g_hHkMouse, code, wParam, lParam);
}

/**
* Initialize DLL: install kbd/mouse hooks.
**/
DLLIMPORT BOOL IdleTrackerInit()
{
	if (g_hHkKeyboard == NULL) {
		g_hHkKeyboard = SetWindowsHookEx(WH_KEYBOARD, KeyboardTracker, g_hInstance, 0);
	}
	if (g_hHkMouse == NULL) {
		g_hHkMouse = SetWindowsHookEx(WH_MOUSE, MouseTracker, g_hInstance, 0);
	}

	g_dwLastTick = GetTickCount(); // init count

	if (!g_hHkKeyboard || !g_hHkMouse)
		return FALSE;
	else
		return TRUE;
}

/**
* Terminate DLL: remove hooks.
**/
DLLIMPORT void IdleTrackerTerm()
{
	BOOL bResult;
	if (g_hHkKeyboard)
	{
		bResult = UnhookWindowsHookEx(g_hHkKeyboard);
		g_hHkKeyboard = NULL;
	}
	if (g_hHkMouse)
	{
		bResult = UnhookWindowsHookEx(g_hHkMouse);
		g_hHkMouse = NULL;
	}
}

/**
* DLL's entry point
**/
BOOL APIENTRY DllMain(HINSTANCE hInstance, DWORD dwReason, LPVOID lpReserved)
{
	switch(dwReason) {
		case DLL_PROCESS_ATTACH:
			DisableThreadLibraryCalls(hInstance);
			g_hInstance = hInstance;
			g_procCount++;
			break;
		case DLL_PROCESS_DETACH:
			//we do an unhook here just in case the user has forgotten.
			g_procCount--;
			if (g_procCount == 0) 
				IdleTrackerTerm();
			break;
	}
	return TRUE;
}

