#ifndef _Beashop_ToolStrip_h_
#define _Beashop_ToolStrip_h_


#include <Core/Core.h>
#include <CtrlCore/CtrlCore.h>
#include <CtrlLib/CtrlLib.h>

using namespace UPP;


class ToolStrip : public FrameCtrl<ParentCtrl>
{
public:
	struct ToolInfo : public Moveable<ToolInfo>
	{
		enum class EType{Ctrl, Seperator};
		
		Ctrl *pCtrl{nullptr};
		Size szPrefered{20, 20};
		EType type{EType::Ctrl};
	};
	
	using TEType = ToolStrip::ToolInfo::EType;
	
	enum class EAlign{LorT, Center, RorB};
	enum class EPosition{Left, Right, Top, Bottom};

	class Seperator : public Ctrl
	{
	public:
		enum class EDirection{Horizental, Vertical};
		static const int SEPERATOR_HEIGHT{8};
		
		void Paint(Draw& w) override
		{
			Ctrl::Paint(w);
			if(visualLine)
			{
				Size sz = GetSize();
				int pos;
				if(dir == EDirection::Horizental)	// draw vertical lines
				{
					pos = (sz.cx - 2) / 2;
					w.DrawLine(pos, 0, pos, sz.cy, 1, SColorShadow());
					pos++;
					w.DrawLine(pos, 0, pos, sz.cy, 1, SColorLight());
				}
				else
				{
					pos = (sz.cy - 2) / 2;
					w.DrawLine(0, pos, sz.cx, pos, 1, SColorShadow());
					pos++;
					w.DrawLine(0, pos, sz.cx, pos, 1, SColorLight());
				}
			}
		}
		
		Size GetMinSize() const override
		{
			return (dir == ToolStrip::Seperator::EDirection::Horizental)?
				Size(SEPERATOR_HEIGHT, 0) : Size(0, SEPERATOR_HEIGHT);
		}
		
		Size GetStdSize() const override
		{
			return (dir == ToolStrip::Seperator::EDirection::Horizental)?
				Size(SEPERATOR_HEIGHT, 20) : Size(20, SEPERATOR_HEIGHT);
		}
		
		Size GetMaxSize() const override
		{
			return (dir == ToolStrip::Seperator::EDirection::Horizental)?
				Size(SEPERATOR_HEIGHT, 20) : Size(20, SEPERATOR_HEIGHT);
		}
		
		Seperator & Dir(EDirection d)
		{
			dir = d;
			return *this;
		}
		
		Seperator & VisualLine(bool b = true)
		{
			visualLine = b;
			return *this;
		}
		
	protected:
		EDirection dir{EDirection::Horizental};	// ToolStrip direction
		bool visualLine{true};
	};
	
	
protected:
	const int BUTTON_OVERFLOW_HEIGHT{10};
	
	Vector<ToolInfo> vTools, vOverflow;
	EAlign align{EAlign::Center};
	EPosition position{EPosition::Top};
	Rect margin{4, 4, 4, 4};
	int gap{2};
	
	Button btnOverflow;
	MenuBar mbOverflow;

	void OnButtonOverflow(void)
	{
		mbOverflow.Clear();
		OverflowBar(mbOverflow);
		mbOverflow.PopUp();
	}
	
	void OverflowBar(Bar &bar)
	{
		for(int i = 0; i < vOverflow.GetCount(); i++)
		{
			ToolInfo &ti = vOverflow[i];
			if(ti.type == TEType::Seperator)
				bar.Separator();
			else
			{
				switch(align)
				{
					case EAlign::LorT:
						ti.pCtrl->LeftPos(0, ti.szPrefered.cx).TopPos(0, ti.szPrefered.cy);
						break;
					case EAlign::RorB:
						ti.pCtrl->RightPos(0, ti.szPrefered.cx).TopPos(0, ti.szPrefered.cy);
						break;
					case EAlign::Center:
						ti.pCtrl->HCenterPos(ti.szPrefered.cx).TopPos(0, ti.szPrefered.cy);
						break;
				}
				bar.Add(*ti.pCtrl, ti.szPrefered);
			}
		}
	}
	
public:
	ToolStrip()
	{
		Add(btnOverflow);
		btnOverflow.Hide();
		btnOverflow << [=]{OnButtonOverflow();};
		mbOverflow.Set([=](Bar &bar){OverflowBar(bar);});
	}
	
	
	~ToolStrip()
	{
		for(auto s : vTools)
		{
			if(s.type == ToolStrip::TEType::Seperator)
				delete s.pCtrl;
		}
	}
	
	
	ToolStrip &AddCtrl(Ctrl &ctrl, int cx = 0, int cy = 0)
	{
		ToolInfo ti;
		ti.pCtrl = &ctrl;
		
		Size sz = ctrl.GetRect().Size();
		sz.cx = sz.cx <= 0? 20 : sz.cx;
		sz.cy = sz.cy <= 0? 20 : sz.cy;
		ti.szPrefered = Size(cx <= 0? sz.cx : cx, cy <= 0? sz.cy : cy);
		ti.type = TEType::Ctrl;
		vTools.Add(ti);
		return *this;
	}
	
	
	ToolStrip &AddSeperator(void)
	{
		ToolInfo ti;
		
		ti.pCtrl = new Seperator();
		ti.type = TEType::Seperator;
		vTools.Add(ti);
		return *this;
	}
	
	
	ToolStrip &Align(ToolStrip::EAlign a)
	{
		align = a;
		return *this;
	}
	
	
	ToolStrip &Margin(int l, int t, int r, int b)
	{
		margin = Rect(l, t, r, b);
		return *this;
	}
	
	
	ToolStrip &Gap(int g)
	{
		gap = g;
		return *this;
	}
	
	ToolStrip & Position(EPosition p)
	{
		position = p;
		return *this;
	}
	
	ToolStrip & PositionLeft(void)	{ return Position(EPosition::Left); }
	ToolStrip & PositionRight(void) { return Position(EPosition::Right); }
	ToolStrip & PositionTop(void) { return Position(EPosition::Top); }
	ToolStrip & PositionBottom(void) { return Position(EPosition::Bottom); }
	
	Size FitSize(void) const
	{
		Size szFit{margin.left + margin.right, margin.top + margin.bottom}, sz{0, 0}, szt;
		if(position == ToolStrip::EPosition::Left ||
			position == ToolStrip::EPosition::Right)	// vertical
		{
			for(auto s : vTools)
			{
				if(s.type == TEType::Ctrl)
				{
					szt = s.szPrefered;
					if(sz.cx < szt.cx)
						sz.cx = szt.cx;
					sz.cy += szt.cy;
				}
				else
				{
					reinterpret_cast<Seperator *>(s.pCtrl)->Dir(Seperator::EDirection::Vertical);
					sz.cy += Seperator::SEPERATOR_HEIGHT;
				}
			}
			if(vTools.GetCount() > 1)
				sz.cy += gap * (vTools.GetCount() - 1);
		}
		else	// horizental
		{
			for(auto s : vTools)
			{
				if(s.type == TEType::Ctrl)
				{
					szt = s.szPrefered;
					if(sz.cy < szt.cy)
						sz.cy = szt.cy;
					sz.cx += szt.cx;
				}
				else
				{
					reinterpret_cast<Seperator *>(s.pCtrl)->Dir(Seperator::EDirection::Horizental);
					sz.cx += Seperator::SEPERATOR_HEIGHT;
				}
			}
			if(vTools.GetCount() > 1)
				sz.cx += gap * (vTools.GetCount() - 1);
		}
		szFit += sz;
		return szFit;
	}

	void FrameLayout(Rect& r) override
	{
		Size sz = FitSize();
		switch(position)
		{
			case EPosition::Left:
				SetFrameRect(r.left, r.top, sz.cx, r.Height());
				r.left += sz.cx;
				break;
			case EPosition::Right:
				SetFrameRect(r.right - sz.cx, r.top, sz.cx, r.Height());
				r.right -= sz.cx;
				break;
			case EPosition::Top:
				SetFrameRect(r.left, r.top, r.Width(), sz.cy);
				r.top += sz.cy;
				break;
			case EPosition::Bottom:
				SetFrameRect(r.left, r.bottom - sz.cy, r.Width(), sz.cy);
				r.bottom -= sz.cy;
				break;
		}
	}
	
	void FrameAddSize(Size& sz) override
	{
		Size fs = FitSize();
		switch(position)
		{
			case EPosition::Left:
			case EPosition::Right:
				sz.cx += fs.cx;
				break;
			case EPosition::Top:
			case EPosition::Bottom:
				sz.cy += fs.cy;
				break;
		}
	}
	
	void FrameAdd(Ctrl& parent) override
	{
		parent.Add(*this);
	}
	
	void FrameRemove() override
	{
		Remove();
	}
	
	Size GetMinSize() const override
	{
		return Size(margin.left + margin.right + BUTTON_OVERFLOW_HEIGHT,
			margin.top + margin.bottom + BUTTON_OVERFLOW_HEIGHT);
	}
	
	Size GetStdSize() const override
	{
		return Size(margin.left + margin.right + BUTTON_OVERFLOW_HEIGHT +
			((position == EPosition::Left || position == EPosition::Right)? 20 : (gap + 20)),
			margin.top + margin.bottom + BUTTON_OVERFLOW_HEIGHT +
			((position == EPosition::Left || position == EPosition::Right)? (gap + 20) : 20));
	}
	
	void Layout() override
	{
		mbOverflow.CloseMenu();
		
		Size sz = GetSize();
		bool bof = false;
		int ofStart = -1;
		int nts, nos = 0;
		nts = (position == EPosition::Left || position == EPosition::Right)?
			sz.cy - margin.top - margin.bottom : sz.cx - margin.left - margin.right;
		for(int i = 0; i < vTools.GetCount(); i++)
		{
			ToolInfo &ti = vTools[i];
			TEType tt = ti.type;
			Size szt = (tt == TEType::Ctrl)? ti.szPrefered :
				Size(Seperator::SEPERATOR_HEIGHT, Seperator::SEPERATOR_HEIGHT);
			Ctrl *parent = ti.pCtrl->GetParent();
			
			if(position == EPosition::Left || position == EPosition::Right)	// strip is vertical
			{
				if(tt == TEType::Seperator)
					szt.cx = sz.cx - margin.left - margin.right;
				if(nos + szt.cy > nts)	// overflow
				{
					bof = true;
					for(int j = i; j >= 0; j--)
					{
						if(nts - nos >= Seperator::SEPERATOR_HEIGHT)
						{
							ofStart = j;
							break;
						}
						else
						{
							nos -= (gap + vTools[j].szPrefered.cy);
						}
					}
					if(ofStart < 0)
						ofStart = 0;
					break;
				}
				if(parent != this)
				{
					ti.pCtrl->Remove();
					Add(*ti.pCtrl);
				}
				switch(align)
				{
					case EAlign::LorT:	// vertical bar, align left
						ti.pCtrl->TopPos(margin.top + nos, szt.cy).LeftPos(margin.left, szt.cx);
						break;
					case EAlign::RorB:
						ti.pCtrl->TopPos(margin.top + nos, szt.cy).RightPos(margin.right, szt.cx);
						break;
					case EAlign::Center:
						ti.pCtrl->TopPos(margin.top + nos, szt.cy).HCenterPos(szt.cx, (margin.left - margin.right) / 2);
						break;
				}
				nos += (szt.cy + gap);
			}
			else	// strip is horizental
			{
				if(tt == TEType::Seperator)
					szt.cy = sz.cy - margin.top - margin.bottom;
				if(nos + szt.cx > nts)
				{
					bof = true;
					for(int j = i; j >= 0; j--)
					{
						if(nts - nos >= Seperator::SEPERATOR_HEIGHT)
						{
							ofStart = j;
							break;
						}
						else
						{
							nos -= (gap + vTools[j].szPrefered.cx);
						}
					}
					if(ofStart < 0)
						ofStart = 0;
					break;
				}
				if(parent != this)
				{
					ti.pCtrl->Remove();
					Add(*ti.pCtrl);
				}
				switch(align)
				{
					case EAlign::LorT:	// horizental bar, align top
						ti.pCtrl->LeftPos(margin.left + nos, szt.cx).TopPos(margin.top, szt.cy);
						break;
					case EAlign::RorB:
						ti.pCtrl->LeftPos(margin.left + nos, szt.cx).BottomPos(margin.bottom, szt.cy);
						break;
					case EAlign::Center:
						ti.pCtrl->LeftPos(margin.left + nos, szt.cx).VCenterPos(szt.cy, (margin.top - margin.bottom) / 2);
						break;
				}
				nos += (szt.cx + gap);
			}
		}
		vOverflow.Clear();
		if(bof)	// tools overflow, show overflow button
		{
			if(position == EPosition::Left || position == EPosition::Right)	// strip is vertical
			{
				btnOverflow.SetImage(CtrlImg::smalldown());
				btnOverflow.BottomPos(margin.bottom, BUTTON_OVERFLOW_HEIGHT).HSizePos(margin.left, margin.right);
//				btnOverflow.TopPos(nos, BUTTON_OVERFLOW_HEIGHT).HSizePos(margin.left, margin.right);
			}
			else
			{
				btnOverflow.SetImage(CtrlImg::smallright());
				btnOverflow.RightPos(margin.right, BUTTON_OVERFLOW_HEIGHT).VSizePos(margin.top, margin.bottom);
//				btnOverflow.LeftPos(nos, BUTTON_OVERFLOW_HEIGHT).VSizePos(margin.top, margin.bottom);
			}
			for(int i = ofStart; i < vTools.GetCount(); i++)
			{
				vTools[i].pCtrl->Remove();
				vOverflow.Add(vTools[i]);
			}
		}
		btnOverflow.Show(bof);
	}
	
	Button & OverflowButton(void) {return btnOverflow;};
};

using TTEType = ToolStrip::TEType;


#endif
