#ifndef _RECT_T_
#define _RECT_T_

#include <Core/Core.h>
#include <CtrlLib/CtrlLib.h>
#include "point_t.h"

using namespace Upp;

template<class T>
struct rect_: Moveable<rect_<T>> {
  T x;
  T y;
  T width;
  T height;

  void Clear() {
    x = y = width = height = 0;
  }

  rect_() {
    x = y = width = height = 0;
  }

  rect_(T x, T y, T width, T height) {
    this->x = x;
    this->y = y;
    this->width = width;
    this->height = height;
  }

  rect_(Rect_<int> b) {
    this->x = b.left;
    this->y = b.top;
    this->width = b.GetWidth();
    this->height = b.GetHeight();
  }

  friend rect_ operator*(rect_ a, T m) {
    return rect_<T>(a.x * m, a.y * m, a.width * m, a.height * m);
  }
  friend rect_ operator/(rect_ a, T m) {
    return rect_<T>(a.x / m, a.y / m, a.width / m, a.height / m);
  }

  point_<T> get_location() {
    return point_<T>(x, y);
  }

  bool intersects(rect_ t) {
    return (
      t.width > 0 && t.height > 0 &&
      width > 0 && height > 0 &&
      t.x < (x + width)  && (t.x + t.width)  > x &&
      t.y < (y + height) && (t.y + t.height) > y
    );
  }

  enum {
    RECT_INSIDE = 0, // 0000
    RECT_LEFT   = 1, // 0001
    RECT_RIGHT  = 2, // 0010
    RECT_BOTTOM = 4, // 0100
    RECT_TOP    = 8, // 1000
  };

  int outcode(double x, double y) {
    int code = RECT_INSIDE;

    if (x < get_x1()) {
      code |= RECT_LEFT;
    } else if (x > get_x2()) {
      code |= RECT_RIGHT;
    }
    if (y < get_y1()) {
      code |= RECT_BOTTOM;
    } else if (y > get_y2()) {
      code |= RECT_TOP;
    }

    return code;
  }

  bool intersects(double& x0, double& y0, double& x1, double& y1) {
    int out0 = outcode(x0, y0);
    int out1 = outcode(x1, y1);
    bool accept = false;

    while (true) {
      if (!(out0 | out1)) {
        accept = true;
        break;
      } else if (out0 & out1) {
        break;
      } else {
        double x, y;
        int outcodeOut = out1 > out0 ? out1 : out0;

        if (outcodeOut & RECT_TOP) {
          x = x0 + (x1 - x0) * (get_y2() - y0) / (y1 - y0);
          y = get_y2();
        } else if (outcodeOut & RECT_BOTTOM) {
          x = x0 + (x1 - x0) * (get_y1() - y0) / (y1 - y0);
          y = get_y1();
        } else if (outcodeOut & RECT_RIGHT) {
          y = y0 + (y1 - y0) * (get_x2() - x0) / (x1 - x0);
          x = get_x2();
        } else if (outcodeOut & RECT_LEFT) {
          y = y0 + (y1 - y0) * (get_x1() - x0) / (x1 - x0);
          x = get_x1();
        }

        if (outcodeOut == out0) {
          x0 = x;
          y0 = y;
          out0 = outcode(x0, y0);
        } else {
          x1 = x;
          y1 = y;
          out1 = outcode(x1, y1);
        }
      }
    }
    return accept;
  }

  T get_x1() {
    return x;
  }

  T get_y1() {
    return y;
  }

  T get_x2() {
    return x + width;
  }

  T get_y2() {
    return y + height;
  }
#if 0
  friend rect_ operator*(rect_ a, rect_<double> b) {
    a.x *= b.x;
    a.y *= b.y;
    a.width *= b.width;
    a.height *= b.height;
  }
#endif
  friend rect_ operator*(rect_ a, rect_<int> b) {
    a.x = (int)(a.x * b.x);
    a.y = (int)(a.y * b.y);
    a.width = (int)(a.width * b.width);
    a.height = (int)(a.height * b.height);
  }

};

typedef rect_<double> rect_f;
typedef rect_<int> rect_i;

#endif //_RECT_T_