#include <CtrlLib/CtrlLib.h>
#include "Xform3D.h"

using namespace Upp;
#define LAYOUTFILE <XForm3DExample/main.lay>
#include <CtrlCore/lay.h>

struct Triangle : Moveable<Triangle> {
	Point3D p1, p2, p3;
	Color c = Color(128, 128, 128);
	Triangle() {}
	Triangle(Point3D a, Point3D b, Point3D c) : p1(a), p2(b), p3(c) {}
	Triangle(double ax, double ay, double az,
	         double bx, double by, double bz,
	         double cx, double cy, double cz)
	         : p1(ax, ay, az)
	         , p2(bx, by, bz)
	         , p3(cx, cy, cz)
	         {
	         }

	friend Triangle operator*(const Triangle& t, const Matrix4D& m) { Triangle q(t.p1 * m, t.p2 * m, t.p3 * m); q.c = t.c; return q; }
};

bool LoadMeshFromFile(const String& path, Vector<Triangle>& mesh)
{
		FileIn fi(path);
		if(!fi)
			return false;
		try {
			Vector<Point3D> v;
			while(!fi.IsEof()) {
				String s = fi.GetLine();
				CParser p(s);
				if(p.IsChar('v')) {
					p.SkipTerm();
					auto& pt = v.Add();
					pt.x = p.ReadDouble();
					pt.y = p.ReadDouble();
					pt.z = p.ReadDouble();
				}
				else
				if(p.IsChar('f')) {
					p.SkipTerm();
					Triangle& t = mesh.Add();
					t.p1 = v[p.ReadInt() - 1];
					t.p2 = v[p.ReadInt() - 1];
					t.p3 = v[p.ReadInt() - 1];
				}
			}
		}
		catch(const CParser::Error& e)
		{
			LOG(e);
			return false;
		}
		return true;
}

struct XForm3DDemo : TopWindow {
	Vector<Triangle> object3d;
	WithPanelLayout<ParentCtrl> panel;
	SplitterFrame sf;
	XForm3DDemo()
	{
		if(!LoadMeshFromFile(GetDataFile("teapot.obj"), object3d))
			return;

		Sizeable().Zoomable().CenterScreen().SetRect(0, 0, 800, 800);
		CtrlLayout(panel);
		AddFrame(sf.Right(panel, 200));
		
		auto refresh = [=] { Refresh(); };
		
		panel.fov.Range(180).Step(1) << refresh;
		
		panel.rx.Range(360).Step(1) << refresh;
		panel.ry.Range(360).Step(1) << refresh;
		panel.rz.Range(360).Step(1) << refresh;

		panel.sx.MinMax(1, 1000).Step(1) << refresh;
		panel.sy.MinMax(1, 1000).Step(1) << refresh;
		panel.sz.MinMax(1, 1000).Step(1) << refresh;

		panel.tx.MinMax(-100, 100).Step(1) << refresh;
		panel.ty.MinMax(-100, 100).Step(1) << refresh;
		panel.tz.MinMax(-1000, -100).Step(1) << refresh;

		panel.fov <<= 45;
		
		panel.rx <<= 180;
		panel.ry <<= 180;
		panel.rz <<= 0;

		panel.sx <<= 1;
		panel.sy <<= 1;
		panel.sz <<= 1;

		panel.tx <<= 0;
		panel.ty <<= 0;
		panel.tz <<= -500;
	}


	void MakeScene(const Vector<Triangle>& mesh, Vector<Triangle>& tris, const Matrix4D& world, const Matrix4D& projection)
	{
		Point3D camera	{ 0, 0, 0 };
		for(Triangle t : mesh) {
			t = t * world;
			Point3D normal = Normal(t.p1, t.p2, t.p3);
			if(DotProduct(t.p1 - camera, normal) < 0) {
				Size sz = GetSize();
				Point3D light_direction(0, 0, -1);
				int c = (int) abs(DotProduct(light_direction.Normalized(), normal) * 255.0);
				t.c = Color(c, c, c);
				t = t * projection;
				t.p1.x += 1.0; t.p1.y += 1.0;
				t.p2.x += 1.0; t.p2.y += 1.0;
				t.p3.x += 1.0; t.p3.y += 1.0;
				t.p1.x *= 0.5 * (double) sz.cx;
				t.p1.y *= 0.5 * (double) sz.cy;
				t.p2.x *= 0.5 * (double) sz.cx;
				t.p2.y *= 0.5 * (double) sz.cy;
				t.p3.x *= 0.5 * (double) sz.cx;
				t.p3.y *= 0.5 * (double) sz.cy;
				tris.AddPick(pick(t));
			}
		}
			
		Sort(tris, [](auto &t1, auto &t2)
		{
			double z1 = (t1.p1.z + t1.p2.z + t1.p3.z) / 3.0;
			double z2 = (t2.p1.z + t2.p2.z + t2.p3.z) / 3.0;
			return abs(z1) < abs(z2);
		});
	}
	
	void Paint(Draw& w) override
	{
		Sizef sz = GetSize();
		
		Matrix4D q = Matrix4D::Identity();
		Matrix4D p = Matrix4D::Identity();
		q *= Matrix4D::Scale(1.0 + double(~panel.sx) * 0.001, 1.0 + double(~panel.sy) * 0.001, 1.0 + double(~panel.sz) * 0.001);
		q *= Matrix4D::RotationX(double(~panel.rx) * M_PI / 180.0);
		q *= Matrix4D::RotationY(double(~panel.ry) * M_PI / 180.0);
		q *= Matrix4D::RotationZ(double(~panel.rz) * M_PI / 180.0);
		q *= Matrix4D::Translation(double(~panel.tx) * 0.1, double(~panel.ty) * 0.1, double(~panel.tz) * 0.1);
		p  = Matrix4D::Perspective(~panel.fov, sz.cy / sz.cx, 0.1, 1000.0);

		Vector<Triangle> mesh;

		MakeScene(object3d, mesh, q, p);

		DrawPainter im(w, sz);
		im.Clear(Black());
		for(const auto& t : mesh) {
			im.Move(t.p1.ToPointf()).Line(t.p2.ToPointf());
			im.Move(t.p2.ToPointf()).Line(t.p3.ToPointf());
			im.Move(t.p3.ToPointf()).Line(t.p1.ToPointf());
			im.Stroke(1, t.c);
			im.Fill(t.c);
		}
	}
};

GUI_APP_MAIN
{
	XForm3DDemo().Run();
}
