/*
 *      Copyright (c) 2024 Scott Duensing, scott@duensing.com
 *
 *      Permission is hereby granted, free of charge, to any person obtaining a copy
 *      of this software and associated documentation files (the "Software"), to deal
 *      in the Software without restriction, including without limitation the rights
 *      to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *      copies of the Software, and to permit persons to whom the Software is
 *      furnished to do so, subject to the following conditions:
 *
 *      The above copyright notice and this permission notice shall be included in
 *      all copies or substantial portions of the Software.
 *
 *      THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *      IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *      FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *      AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *      LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *      OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 *      SOFTWARE.
 */


#include "StoreManager.h"


// --- CONFIG ----------------------------------------------------------------


StoreManager::Config::Config()
	: intUseSSH(0)
	, strSSHServer("store.example.com")
	, intSSHPort(22)
	, strSSHAuth(SSH_AUTH_AGENT)
	, strSSHUser("user")
	, strSSHPass("password")
	, strSSHPrivateKey("~/.ssh/id_rsa")
	, strSSHPublicKey("~/.ssh/id_rsa.pub")
	, strDBConnection(DB_CONN_HOST_PORT)
	, strDBServer("db.example.com")
	, intDBPort(3306)
	, strDBUser("user")
	, strDBPass("password")
	, strDBDatabase("designs")
{
}


String StoreManager::Config::GetConfigFile() {
	return ConfigFile(CONFIG_FILE);
}


void StoreManager::Config::Jsonize(JsonIO& json) {
	json("UseSSH",        intUseSSH)
	    ("SSHServer",     strSSHServer)
	    ("SSHPort",       intSSHPort)
	    ("SSHAuth",       strSSHAuth)
	    ("SSHUser",       strSSHUser)
	    ("SSHPass",       strSSHPass)
	    ("SSHPrivateKey", strSSHPrivateKey)
	    ("SSHPublicKey",  strSSHPublicKey)
	    ("DBConnection",  strDBConnection)
	    ("DBServer",      strDBServer)
	    ("DBPort",        intDBPort)
	    ("DBUser",        strDBUser)
	    ("DBPass",        strDBPass)
	    ("DBDatabase",    strDBDatabase);
}


// --- STOREMANAGER -----------------------------------------------------------


StoreManager::StoreManager() {
	// Load config.
	LoadFromJsonFile(settings, settings.GetConfigFile());

	// Load main window layout.
	CtrlLayout(*this, String(APP_NAME));
	
	// Add menubar.
	AddFrame(menu);
	menu.Set([=](Bar& bar) { MainMenuBar(bar); });
}


bool StoreManager::ConnectToDB(Config& config) {
	bool success = false;
	
	// Do they want SSH?
	if (1 == config.intUseSSH) {
		sshSession.Timeout(30000);
		// What kind of authentication?
		if (SSH_AUTH_AGENT == config.strSSHAuth) {
			sshSession.AgentAuth();
		}
		if (SSH_AUTH_KEY == config.strSSHAuth) {
			sshSession.PublicKeyAuth().Keys(config.strSSHPrivateKey, config.strSSHPublicKey, config.strSSHPass, true);
		}
		// Try to connect.
		if (!sshSession.Connect(config.strSSHServer, config.intSSHPort, config.strSSHUser, config.strSSHPass)) {
			PromptOK(sshSession.GetErrorDesc());
			return false;
		}
	}
	
	// Connect to DB.
	if (DB_CONN_HOST_PORT == config.strDBConnection) {
		success = sqlSession.Connect(config.strDBUser, config.strDBPass, config.strDBDatabase, config.strDBServer, config.intDBPort);
		if (!success) PromptOK(sqlSession.GetLastError());

	}
	
	return success;
}


void StoreManager::CopyDBSettingsToConfig(WithDatabaseSettingsLayout<TopWindow>& dialog, Config& config) {
	config.intUseSSH        = ~dialog.oBoxUseSSH;
	config.strSSHServer     = ~dialog.eStrSSHServer;
	config.intSSHPort       = ~dialog.eIntSSHPort;
	config.strSSHAuth       = ~dialog.lstSSHAuth;
	config.strSSHUser       = ~dialog.eStrSSHUser;
	config.strSSHPass       = ~dialog.eStrSSHPass;
	config.strSSHPrivateKey = ~dialog.eStrSSHPrivateKey;
	config.strSSHPublicKey  = ~dialog.eStrSSHPublicKey;
	
	config.strDBConnection = ~dialog.lstDBConnection;
	config.strDBServer     = ~dialog.eStrDBServer;
	config.intDBPort       = ~dialog.eIntDBPort;
	config.strDBUser       = ~dialog.eStrDBUser;
	config.strDBPass       = ~dialog.eStrDBPass;
	config.strDBDatabase   = ~dialog.eStrDBDatabase;
}


void StoreManager::DisconnectFromDB(Config& config) {
	// Do they want SSH?
	if (1 == config.intUseSSH) {
		// Disconnect.
		sshSession.Disconnect();
	}
	
	sqlSession.Close();
}


void StoreManager::FileMenu(Bar& bar) {
	bar.Add("Exit", [=] {
		if (PromptOKCancel("Exit " << String(APP_NAME) << "?"))
			Break();
	});
}


void StoreManager::MainMenuBar(Bar& bar) {
   bar.Sub("File", [=](Bar& bar) { FileMenu(bar); });
   bar.Sub("Settings", [=](Bar& bar) { SettingsMenu(bar); });
}


void StoreManager::SettingsMenu(Bar& bar) {
	bar.Add("Database...", [=] { ShowDBSettings(); });
}


void StoreManager::ShowDBSettings() {
	WithDatabaseSettingsLayout<TopWindow> dbSettings;
	CtrlLayoutOKCancel(dbSettings, "Database Settings");

	// Set up dialog.
	dbSettings.lstSSHAuth.Add(SSH_AUTH_KEY);
	dbSettings.lstSSHAuth.Add(SSH_AUTH_AGENT);
	dbSettings.eStrSSHPass.Password();
	
	dbSettings.lstDBConnection.Add(DB_CONN_HOST_PORT);
	dbSettings.eStrDBPass.Password();

	dbSettings.oBoxUseSSH.EnableBox(settings.intUseSSH);

	// Display settings.
	dbSettings.oBoxUseSSH        <<= settings.intUseSSH;
	dbSettings.eStrSSHServer     <<= settings.strSSHServer;
	dbSettings.eIntSSHPort       <<= settings.intSSHPort;
	dbSettings.lstSSHAuth        <<= settings.strSSHAuth;
	dbSettings.eStrSSHUser       <<= settings.strSSHUser;
	dbSettings.eStrSSHPass       <<= settings.strSSHPass;
	dbSettings.eStrSSHPrivateKey <<= settings.strSSHPrivateKey;
	dbSettings.eStrSSHPublicKey  <<= settings.strSSHPublicKey;

	dbSettings.lstDBConnection <<= settings.strDBConnection;
	dbSettings.eStrDBServer    <<= settings.strDBServer;
	dbSettings.eIntDBPort      <<= settings.intDBPort;
	dbSettings.eStrDBUser      <<= settings.strDBUser;
	dbSettings.eStrDBPass      <<= settings.strDBPass;
	dbSettings.eStrDBDatabase  <<= settings.strDBDatabase;

	UpdateDBSettingsFields(dbSettings);

	// Did they toggle SSH?
	dbSettings.oBoxUseSSH << [&] {
		// We're not using AutoBox because it changes fields after
		// this event, not before.
		dbSettings.oBoxUseSSH.EnableBox(~dbSettings.oBoxUseSSH);
		if (1 == ~dbSettings.oBoxUseSSH) UpdateDBSettingsFields(dbSettings);
	};
	
	// Did they change auth types?
	dbSettings.lstSSHAuth << [&] { UpdateDBSettingsFields(dbSettings); };
	
	// Did they press TEST?
	dbSettings.btnTest << [&] {
		Config testSettings;
		CopyDBSettingsToConfig(dbSettings, testSettings);
		if (ConnectToDB(testSettings)) PromptOK("Connection successful!");
		DisconnectFromDB(testSettings);
	};
	
	// Did they press BROWSE?
	dbSettings.btnSSHPrivateKey << [&] {
		FileSel browse;
		if (browse.ExecuteOpen("Select Private Key File")) {
			dbSettings.eStrSSHPrivateKey <<= ~browse;
		}
	};
	dbSettings.btnSSHPublicKey << [&] {
		FileSel browse;
		if (browse.ExecuteOpen("Select Public Key File")) {
			dbSettings.eStrSSHPublicKey <<= ~browse;
		}
	};

	// Run dialog.
	if (dbSettings.Execute() != IDOK) return;
	
	// Save settings.
	CopyDBSettingsToConfig(dbSettings, settings);
	StoreAsJsonFile(settings, settings.GetConfigFile(), true);
}


void StoreManager::UpdateDBSettingsFields(WithDatabaseSettingsLayout<TopWindow>& dialog) {
	// Only do SSH fields if SSH is enabled.
	if (1 == ~dialog.oBoxUseSSH) {
		// Disable dynamic fields.
		dialog.eStrSSHUser.Disable();
		dialog.eStrSSHPass.Disable();
		dialog.eStrSSHPrivateKey.Disable();
		dialog.btnSSHPrivateKey.Disable();
		dialog.eStrSSHPublicKey.Disable();
		dialog.btnSSHPublicKey.Disable();
		
		// What kind of authentication?
		if (SSH_AUTH_AGENT == ~dialog.lstSSHAuth) {
			dialog.eStrSSHUser.Enable();
		}
		if (SSH_AUTH_KEY == ~dialog.lstSSHAuth) {
			dialog.eStrSSHPass.Enable();
			dialog.eStrSSHPrivateKey.Enable();
			dialog.btnSSHPrivateKey.Enable();
			dialog.eStrSSHPublicKey.Enable();
			dialog.btnSSHPublicKey.Enable();
		}
	}
}


GUI_APP_MAIN {
	SetAppName(APP_NAME);
	StoreManager().Run();
}
