#!/usr/bin/env bash
set -euo pipefail

OPENSSL_VER=3.6.0
SRC_DIR="openssl-${OPENSSL_VER}"
INSTALL_ROOT="$HOME/openssl"
LLVM_MINGW="/opt/llvm-mingw"

# Prepare output dirs
mkdir -p "$INSTALL_ROOT/x86" "$INSTALL_ROOT/x86_64" "$INSTALL_ROOT/aarch64" "$INSTALL_ROOT/include"

cd "$SRC_DIR"

build_arch() {
  local arch_name="$1"
  local target_conf="$2"
  local xc_prefix="$3"

  echo "=== Building for $arch_name ==="
  make clean || true

  export CC="clang"
  export AR="llvm-ar"
  export RANLIB="llvm-ranlib"
  export PATH="$LLVM_MINGW/bin:$PATH"

  # Disable asm/tests to reduce cross‑compile failures
  perl Configure "$target_conf" no-shared no-asm no-tests \
    --cross-compile-prefix="$xc_prefix" \
    --prefix="$(pwd)/install-$arch_name"

  # Build; tolerate failures but try to produce archives
  make -j"$(nproc)" || true

  # Copy archives if present
  for lib in libcrypto libssl; do
    if [ -f "$lib.a" ]; then
      cp "$lib.a" "$INSTALL_ROOT/$arch_name/"
    else
      echo "Warning: $lib.a not found for $arch_name"
    fi
  done
}

# Build x86 (32-bit Windows) with llvm-mingw
build_arch "x86" "mingw" "$LLVM_MINGW/bin/i686-w64-mingw32-"

# Build x86_64 (64-bit Windows) with llvm-mingw
build_arch "x86_64" "mingw64" "$LLVM_MINGW/bin/x86_64-w64-mingw32-"

# Build ARM64 (Windows on ARM) with llvm-mingw
build_arch "aarch64" "mingwarm64" "$LLVM_MINGW/bin/aarch64-w64-mingw32-"

# Copy headers once
rm -rf "$INSTALL_ROOT/include/openssl"
cp -r include/openssl "$INSTALL_ROOT/include/"

echo "All builds attempted. Archives are under $INSTALL_ROOT/{x86,x86_64,aarch64}, headers under $INSTALL_ROOT/include"

