#include	<Core/Core.h>

#include "PackCascade.h"

////////////////////////////////////////////////////////////////////////////////////////////////
// GETS OPENCASCADE ROOT PATH
String GetCasRoot(void)
{
	String RootVar("CASROOT");
	String CasRoot;
	int i = Environment().Find(RootVar);
	if(i >= 0)
		CasRoot = Environment()[i];
	else
		CasRoot = "/opt/OpenCASCADE6.2.0/ros";
	if(!DirectoryExists(CasRoot))
		CasRoot = "";
	return CasRoot;
	
} // END getCasRoot()

////////////////////////////////////////////////////////////////////////////////////////////////
// DIVIDES A CASCADE FILE NAME LIKE xxxxxx_yyyyy.zzz
// INTO 2 PARTS :
//   xxxxxxx
//   yyyyyyy.zzz
// IF NAME DOESN'T HAVE THE '_' CHARACTER, THE FIRST PART
// IS KEPT EQUAL TO THE SECOND :
//   abcdef.hpp --> abcdef + abcdef.hpp
// ERROR IF NAME DOESN'T CONTAIN NEITHER '_' NOR '.'
// SPECIAL CASE FOR NAMES STARTING WTH Handle_
bool SplitCascadeName(String const &Name, String &First, String &Second)
{
	int pos;
	
	if((pos = Name.Find("_")) < 0)
	{
		if((pos = Name.Find(".")) < 0)
			return false;
		First = Name.Mid(0, pos);
		Second = Name;
	}
	else
	{
		First = Name.Mid(0, pos);
		Second = Name.Mid(pos+1);
		if(First == "Handle")
		{
			String F2, S2;
			SplitCascadeName(Second, F2, S2);
			First = F2;
			Second = "Handle_" + S2;
		}
		else
		{
			String F2, S2;
			SplitCascadeName(Second, F2, S2);
			if(First == F2)
				Second = First + "_" + Second;
		}
	}
	if(First == "" || Second == "" || isspace(First[0]) || isspace(Second[0]))
		return false;
	return true;
	
} // END SplitCascadeName()

////////////////////////////////////////////////////////////////////////////////////////////////
// GETS DIRECTORY CONTENT
Array<String> GetDirectoryContent(const String &Path, bool Folders)
{
	
	// GETS INCLUDE FOLDER CONTENTS
	Array<FileSystemInfo::FileInfo> DirFiles;
	DirFiles = StdFileSystemInfo().Find(Path);
	
	// COPIES ALL NAMES INSIDE AN ARRAY
	Array<String> DirArray;
	for(int i = 0 ; i < DirFiles.GetCount() ; i++)
	{
		if( (Folders && DirFiles[i].is_folder) || (!Folders &&  !DirFiles[i].is_folder) && DirFiles[i].filename != "." && DirFiles[i].filename != "..")
		  DirArray << DirFiles[i].filename;
	}
	
	return DirArray;

} // END GetDirectoryContent()

////////////////////////////////////////////////////////////////////////////////////////////////
// GETS ALL THE INCLUDE FILES FROM OPENCASCADE 'inc' DIR
Array<String> GetIncludeNames(const String &CasRoot)
{
	// GETS INCLUDE FOLDER CONTENTS
	Array<String> IncArray;
	IncArray = GetDirectoryContent(CasRoot + "/inc/*.*", false);
	
	return IncArray;

} // END GetIncludeNames()

////////////////////////////////////////////////////////////////////////////////////////////////
// FROM THE ARRAY OF INCLUDE FILE NAMES, GETS THE ARRAY OF PACKAGE NAMES
Array<String> GetPackageNames(const Array<String> &IncArray)
{
	Array<String> PackageNames;
	String First, Second;
	for(int i = 0 ; i < IncArray.GetCount() ; i++)
	{
		if(!SplitCascadeName(IncArray[i], First, Second))
			continue;
		if(FindIndex(PackageNames, First) < 0)
			PackageNames << First;
	}
	
	return PackageNames;
	
} // END GetPackageNames()

////////////////////////////////////////////////////////////////////////////////////////////////
// FROM DEST PATH GETS TEMPLATE FILE NAMES
Array<String> GetTemplateNames(const String &DestPath)
{
	// GETS ALL TEMPLATE FILE NAMES ON DEST PATH
	// EXTRACT TEMPLATE NAMES FROM FILENAMES
	Array<String> TemplateFileNames = GetDirectoryContent(DestPath + "/*.template", false);
	String TemplateName;
	Array<String> TemplateNames;
	for(int i = 0 ; i < TemplateFileNames.GetCount() ; i++)
	{
		TemplateName = TemplateFileNames[i];
		int p = TemplateName.Find(".");
		if(p == -1)
			continue;
		TemplateNames << (TemplateName.Mid(0, p));
	}
	
	return TemplateNames;
	
} // END GetTemplateNames()

////////////////////////////////////////////////////////////////////////////////////////////////
// READS A TEMPLATE FILE, RETURNING AN ARRAY OF PACKAGE NAMES
Array<String> ReadTemplate(const String &TemplatePath)
{
	Array<String> TemplateLines;
	String Line;
	
	FileIn f(TemplatePath);
	while(!f.IsEof())
	{
		Line = f.GetLine();
		if(Line == "" || isspace(Line[0]))
			continue;
		TemplateLines << Line;
	}
	
	return TemplateLines;
	
} // END ReadTemplate()

////////////////////////////////////////////////////////////////////////////////////////////////
// COPIES A FILE PATCHING #INCLUDE DIRECTIVES INSIDE
// IT DOES A LIMITATE CHECK ON FILE, SO IT' NOT ERROR-FREE
// IF SOME STRANGE CODE IS USED
bool CopyPatch(const String &CasRoot, const String &Source, const String &Dest, const Index<String> &Packages, const Array<String> &Categories)
{
	FileIn fIn(Source);
	FileOut fOut(Dest);
	String Line;
	char Delimiter;
	int IncludeStart, IncludeLen;
	String IncludeName;
	String First, Second;
	int iPackage;
	
	enum { none, preproc, inString, inComment, end } State;
	
	State = none;
	while(!fIn.IsEof())
	{
		Line = fIn.GetLine();
		
		int i = 0 ;
		while(i < Line.GetCount())
		{
			switch(State)
			{
				// NORMAL STATE, LOOKS FOR ALL
				case none :
					if(Line[i] == '#')
						State = preproc;
					else if(Line[i] == '"')
						State = inString;
					else if(Line[i] == 0x5c)
						i++;
					else if(Line[i] == '/' && i < Line.GetCount()-1)
					{
						// C++ STYLE COMMENT ?
						if(Line[i+1] == '/')
						{
							// YES, SKIP REST OF LINE...
							i = Line.GetCount();
						}
						// C STYLE COMMENT ?
						else if(Line[i+1] == '*')
						{
							// YES, CHANGE STATE TO inComment
							// AND GO LOOK FOR END OF COMMENT
							State = inComment;
							i++;
						}
					}
					break;

				// INSIDE STRING STATE, LOOKS ONLY FOR END OF STRING
				// OR END OF LINE, WHCH TERMINATES THE STRING TOO
				case inString :
					if(Line[i] == '"' || i >= Line.GetCount()-1)
						State = none;
					else if(Line[i] == 0x5c)
						i++;
					break;
					
				// INSIDE C-STYLE COMMENT, LOOKS ONLY FOR END OF COMMENT
				case inComment :
					if(i < Line.GetCount() -1 && Line[i] == '*' && Line[i+1] == '/')
					{
						State = none;
						i++;
					}
					break;
					
				// PREPROCESSOR DIRECTIVE FOUND, LOOK IF IT'S #include
				case preproc :

					// RESETS STATE FOR FOLLOWING OPERATIONE
					State = none;
					
					// SKIP BLANKS....
					while(i < Line.GetCount() && isspace(Line[i]))
						i++;
					
					if(i < Line.GetCount() - 7 && Line.Mid(i, 7) == "include")
					{
						// GO TO THE END OF DIRECTIVE
						i += 7;
						
						// SKIP BLANKS
						while(i < Line.GetCount() && isspace(Line[i]))
							i++;
						
						// IF END OF LINE, ERROR IN DIRECTIVE, SO SKIP IT
						if(i >= Line.GetCount()-2) // 2 is space for 2 delimiters
						{
							i = Line.GetCount();
							continue;
						}
						
						// GETS INCLUDE DELIMITER AND START POS IF INCLUDE NAME
						Delimiter = Line[i++];
						if(Delimiter == '<')
							Delimiter = '>';
						IncludeStart = i;
						
						// LOOKS FORWARD FOR CLOSING DELIMITER
						while(i < Line.GetCount() && Line[i] != Delimiter)
							i++;
						
						// IF END OF LINE, ERROR IN DIRECTIVE, SO SKIP IT
						if(i >= Line.GetCount())
							continue;
						
						// SAVES INCLUDE NAME LENGTH
						IncludeLen = i - IncludeStart;
						
						// LOOKS IF INCLUDE FILE IS AN OPENCASCADE PACKAGE
						IncludeName = Line.Mid(IncludeStart, IncludeLen);
						if(!SplitCascadeName(IncludeName, First, Second))
							continue;
						if((iPackage = Packages.Find(First)) >= 0)
						{
							// SHOULD BE AN OPENCASCADE ONE
							// LET'S CHECK IF FILE EXISTS... AND WHERE IS IT
							// AND BUILD CORRECT DEST PATH
							if(FileExists(CasRoot + "/inc/" + IncludeName))
								IncludeName = Categories[iPackage] + "/" + Packages[iPackage] + "/" + Second;
							else if(FileExists(CasRoot + "/drv/" + Packages[iPackage] + "/"  + IncludeName))
								IncludeName = Categories[iPackage] + "/" + Packages[iPackage] + "/drv/" + Second;
							else if(FileExists(CasRoot + "/src/" + Packages[iPackage] + "/"  + IncludeName))
								IncludeName = Categories[iPackage] + "/" + Packages[iPackage] + "/src/" + Second;
							else
								continue;
							
							// SUBSTITUTE THE CORRECT INCLUDE IN SOURCE LINE
							Line = Line.Mid(0, IncludeStart-1) + "<" + IncludeName + ">" + Line.Mid(IncludeStart+IncludeLen+2);
							
							// ADJUST THE LINE INDEX
							i = IncludeStart + IncludeName.GetCount() + 1;
						}
					} // if #include
					
					// HERE a DIRTY TRICK TO DEAL WITH #includes REFERRING TO #defined FILES..
					// IT REPLaCES ONLY THE VALUE OF THE MACRO IF AND ONLY IF IT' S SURROUNDED
					// BY <> AND IS PART OF OpenCascade THREE
					// DON'T DEAL WITH INCLUDES WITH '"' DELIMITER... FOR NOW
					if(i < Line.GetCount() - 6 && Line.Mid(i, 6) == "define")
					{
						// GO TO THE END OF DIRECTIVE
						i += 7;
						
						// SKIP BLANKS
						while(i < Line.GetCount() && isspace(Line[i]))
							i++;
						
						// SEARCH FOR < DELIMITER
						while(i < Line.GetCount() && Line[i] != '<')
							i++;
						
						// IF END OF LINE, NOT FOUND, SO SKIP IT
						if(i >= Line.GetCount())
							continue;
						
						// GETS START OF INCLUDE NAME
						i++;
						IncludeStart = i;
						
						// LOOKS FORWARD FOR CLOSING DELIMITER
						while(i < Line.GetCount() && Line[i] != '>')
							i++;
						
						// IF END OF LINE, ERROR IN DIRECTIVE, SO SKIP IT
						if(i >= Line.GetCount())
							continue;
						
						// SAVES INCLUDE NAME LENGTH
						IncludeLen = i - IncludeStart;
						
						// LOOKS IF INCLUDE FILE IS AN OPENCASCADE PACKAGE
						IncludeName = Line.Mid(IncludeStart, IncludeLen);
						if(!SplitCascadeName(IncludeName, First, Second))
							continue;
						if((iPackage = Packages.Find(First)) >= 0)
						{
							// SHOULD BE AN OPENCASCADE ONE
							// LET'S CHECK IF FILE EXISTS... AND WHERE IS IT
							// AND BUILD CORRECT DEST PATH
							if(FileExists(CasRoot + "/inc/" + IncludeName))
								IncludeName = Categories[iPackage] + "/" + Packages[iPackage] + "/" + Second;
							else if(FileExists(CasRoot + "/drv/" + Packages[iPackage] + "/"  + IncludeName))
								IncludeName = Categories[iPackage] + "/" + Packages[iPackage] + "/drv/" + Second;
							else if(FileExists(CasRoot + "/src/" + Packages[iPackage] + "/"  + IncludeName))
								IncludeName = Categories[iPackage] + "/" + Packages[iPackage] + "/src/" + Second;
							else
								continue;

							// SUBSTITUTE THE CORRECT INCLUDE IN SOURCE LINE
							Line = Line.Mid(0, IncludeStart-1) + "<" + IncludeName + ">" + Line.Mid(IncludeStart+IncludeLen+2);
							
							// ADJUST THE LINE INDEX
							i = IncludeStart + IncludeName.GetCount() + 1;
						}
					} // if #define
							
					break;
					
			} // switch
			i++;
		}

		fOut.PutLine(Line);
	}
	return true;

} // END CopyPatch()



////////////////////////////////////////////////////////////////////////////////////////////////
// PACKS OPENCASCADE FOR UPP USAGE
bool PackageCascade(const String &CasRoot, const String &DestPath)
{
	// GETS THE TEMPLATE NAMES FROM DESTINATION DIRECTORY
	// (MUST EXIST AND FILLED WITH INCLUDE FILE NAMES)
	Array<String> TemplateNames = GetTemplateNames(DestPath);

	// FOR EACH TEMPLATE, EXTRACTS INCLUDED PACKAGE NAMES
	// GETTING THEM FROM INCLUDE FILE NAMES AND MAPS THEM TO TEMPLATE NAME
	Index<String> Packages;
	Array<String> Categories;
	for(int i = 0 ; i < TemplateNames.GetCount() ; i++)
	{
		// READS THE TEMPLATE FILE
		Array<String> TemplateLines = ReadTemplate(DestPath + "/" + TemplateNames[i] + ".template");
		
		// EXTRACTS THE PACKAGE NAMES
		Array<String> TemplatePackages = GetPackageNames(TemplateLines);
		
		// MAPS THE PACKAGE NAMES WITH CORRESPONDING TEMPLATE
		// AND FILLS THE GLOBAL PACKAGES ARRAY FROM TEMPLATES
		for(int j = 0 ; j < TemplatePackages.GetCount() ; j++)
		{
			Packages.Add(TemplatePackages[j]);
			Categories.Add(TemplateNames[i]);
		}
	}
	
	// HERE WE MUST DEAL WITH INCLUDE FILES THAT ARE NOT INCLUDED IN PACKAGES
	// INSIDE THE TEMPLATES; SHOULD NOT HAPPEN, BUT... WE MAP THEM TO 'Others' TEMPLATE
	
	// FIRST, WE READ THE INCLUDE FILES....
	Array<String> IncludeFiles = GetIncludeNames(CasRoot);
	
	// ...FROM THEM WE EXTRACT A PACKAGE NAMES LIST....
	Array<String> PackagesFromIncludes = GetPackageNames(IncludeFiles);
	
	// AND WE LOOK FOR PACKAGES NOT INCLUDED IN TEMPLATES
	// MAPPING THEM TO 'Others' CATEGORY
	bool FoundOrphan = false;
	for(int i = 0 ; i < PackagesFromIncludes.GetCount() ; i++)
	{
		if(Packages.Find(PackagesFromIncludes[i]) < 0)
		{
			Packages.Add(PackagesFromIncludes[i]);
			Categories.Add("Others");
			FoundOrphan = true;
		}
	}
	if(FoundOrphan)
		TemplateNames << "Others";
	
	////////////////////////////////////////////////////////////////////////////////////////////
	// HERE WE HAVE :
	//  All template (category) names in TemplateNames
	//  2 Arrays mapping  Package names with corresponding Templates in Packages+Categories
	//  the complete array of include file names in IncludeFiles
	//
	// We must now create package folders on destination path
	// and copy include files there, changing #include directives if necessary
	
	// FIRST, WE CREATE THE CATEGORY PATHS...
	String ErrorStr;
	String FullPath;
	String SrcPath, DrvPath, OthersPath;
	for(int i = 0 ; i < TemplateNames.GetCount() ; i++)
	{
		FullPath = DestPath + "/" + TemplateNames[i];
		if(!StdFileSystemInfo().FolderExists(FullPath))
		{
			if(!StdFileSystemInfo().CreateFolder(FullPath, ErrorStr))
			{
				Cout() << "Error '" << ErrorStr << " creating '" << FullPath << "'\n";
				return false;
			}
		}
	}
	
	// THEN THE PACKAGES PATHS.....
	for(int i = 0 ; i < Packages.GetCount() ; i++)
	{
		FullPath = DestPath + "/" + Categories[i] + "/" + Packages[i];
		if(!StdFileSystemInfo().FolderExists(FullPath))
		{
			if(!StdFileSystemInfo().CreateFolder(FullPath, ErrorStr))
			{
				Cout() << "Error '" << ErrorStr << " creating '" << FullPath << "'\n";
				return false;
			}
		}
	}
	
	// NOW WE MUST COPY ALL INCLUDE FILES, PATCHING THE INCLUDE DIRECTIVES
	for(int i = 0 ; i < IncludeFiles.GetCount() ; i++)
	{
		String First, Second;
		if(!SplitCascadeName(IncludeFiles[i], First, Second))
		  continue;
		String Source = CasRoot + "/inc/" + IncludeFiles[i];
		int iPackage = Packages.Find(First);
		String Category = Categories[iPackage];
		String Dest = DestPath + "/" + Category + "/" + First + "/" + Second;
		CopyPatch(CasRoot, Source, Dest, Packages, Categories);
	}

	return true;

} // END PackageCascade()

////////////////////////////////////////////////////////////////////////////////////////////////
CONSOLE_APP_MAIN
{
	int argc = CommandLine().GetCount();
	const Vector<String>& argv = CommandLine();
	String CasRoot;

	if(argc == 1)
	{
		// CHECKS FOR DESTINATION FOLDER
		if(!FileSystemInfo().FolderExists(argv[0]))
		{
			Cout() << "Destination folder '" << argv[0] << "' don't exist";
			exit(1);
		}
		
		// CHECKS FOR EXISTING OPENCASCADE
		CasRoot = GetCasRoot();
		if(CasRoot == "")
		{
			Cout() << "OpenCASCADE folder not found\nPlease check OpenCASCADE setup\n";
			exit(1);
		}

		PackageCascade(CasRoot, argv[0]);
		SetExitCode(0);
	}
	else
	{
		Cout() << "USAGE : PackCascade <dest root path>\n";
		exit(1);
	}
}
