#include	<Core/Core.h>

#include "PackCascade.h"

// OPTIONS PER PACKAGE AND PER FILE
//#define PACKAGE_OPTIONS		"-ffunction-sections  -fPIC -funsigned-char -Wall -fmessage-length=0 -DLIN -DLININTEL -DOCC_CONVERT_SIGNALS  -DNOPROTECTION -DCSFDB -DHAVE_WOK_CONFIG_H  -DHAVE_CONFIG_H -DNDEBUG -DNo_Exception -MD"
//#define CXX_OPTIONS			"-ffriend-injection -fpermissive"
#define PACKAGE_OPTIONS		"-ffunction-sections  -fPIC -funsigned-char -Wall -fmessage-length=0 -DLIN -DLININTEL -DOCC_CONVERT_SIGNALS  -DNOPROTECTION -DCSFDB -DHAVE_WOK_CONFIG_H  -DHAVE_CONFIG_H -DNDEBUG -DNo_Exception -MD -ffriend-injection -fpermissive"
#define CXX_OPTIONS			""
#define C_OPTIONS			""


/*
////////////////////////////////////////////////////////////////////////////////////////////////
// FILE TYPES BY EXTENSIONS
// NEEDED TO DIVIDE FILES INTO SUBDIRS AND OTHERS
typedef enum {other, inc, drv, src} FileTypes;

////////////////////////////////////////////////////////////////////////////////////////////////
// FROM FILE NAME, GETS THE OPENCASCADE FILE TYPE
FileTypes GetFileType(String const &Name)
{
	Index<String> SrcExt, IncExt, DrvExt;
	SrcExt << ".c" << ".cxx" << ".cpp" << ".pxx";
	IncExt << ".h" << ".hxx" << ".lxx";
	DrvExt << ".ixx" << ".jxx" << ".gxx";

	String Ext = GetFileExt(Name);
	if(SrcExt.Find(Ext) >= 0)
		return src;
	else if(IncExt.Find(Ext) >= 0)
		return inc;
	else if(DrvExt.Find(Ext) >= 0)
		return drv;
	else
		return other;

} // END GetFileType()
*/

////////////////////////////////////////////////////////////////////////////////////////////////
// GETS OPENCASCADE ROOT PATH
String GetCasRoot(void)
{
	String RootVar("CASROOT");
	String CasRoot;
	int i = Environment().Find(RootVar);
	if(i >= 0)
		CasRoot = Environment()[i];
	else
		CasRoot = "/opt/OpenCASCADE6.2.0/ros";
	if(!DirectoryExists(CasRoot))
		CasRoot = "";
	return CasRoot;
	
} // END getCasRoot()

////////////////////////////////////////////////////////////////////////////////////////////////
// DIVIDES A CASCADE FILE NAME LIKE xxxxxx_yyyyy.zzz
// INTO 2 PARTS :
//   xxxxxxx
//   yyyyyyy.zzz
// IF NAME DOESN'T HAVE THE '_' CHARACTER, THE FIRST PART
// IS KEPT EQUAL TO THE SECOND :
//   abcdef.hpp --> abcdef + abcdef.hpp
// ERROR IF NAME DOESN'T CONTAIN NEITHER '_' NOR '.'
// SPECIAL CASE FOR NAMES STARTING WTH Handle_
bool SplitCascadeName(String const &Name, String &First, String &Second)
{
	int pos;
	
	if((pos = Name.Find("_")) < 0)
	{
		if((pos = Name.Find(".")) < 0)
			return false;
		First = Name.Mid(0, pos);
		Second = "_" + Name;
	}
	else
	{
		First = Name.Mid(0, pos);
		Second = Name.Mid(pos+1);
		if(First == "Handle")
		{
			String F2, S2;
			SplitCascadeName(Second, F2, S2);
			First = F2;
			Second = "Handle_" + S2;
		}
	}
	if(First == "" || Second == "" || isspace(First[0]) || isspace(Second[0]))
		return false;
	return true;
	
} // END SplitCascadeName()

/*
////////////////////////////////////////////////////////////////////////////////////////////////
// THE OPPOSITE OF SplitCascadeName()
String JoinCascadeName(const String &First, const String &Second)
{
	String Name;
	if(Second.Mid(0, 7) == "Handle_")
		Name = "Handle_" + First + "_" + Second.Mid(7);
	else
		Name += First + "_" + Second;
	return Name;
	
} // END JoinCascadeName()
*/

////////////////////////////////////////////////////////////////////////////////////////////////
// GETS DIRECTORY CONTENT
Array<String> GetDirectoryContent(const String &Path, bool Folders)
{
	
	// GETS INCLUDE FOLDER CONTENTS
	Array<FileSystemInfo::FileInfo> DirFiles;
	DirFiles = StdFileSystemInfo().Find(Path);
	
	// COPIES ALL NAMES INSIDE AN ARRAY
	Array<String> DirArray;
	for(int i = 0 ; i < DirFiles.GetCount() ; i++)
	{
		if( (Folders && DirFiles[i].is_folder) || (!Folders &&  !DirFiles[i].is_folder) && DirFiles[i].filename != "." && DirFiles[i].filename != "..")
		  DirArray << DirFiles[i].filename;
	}
	
	return DirArray;

} // END GetDirectoryContent()

////////////////////////////////////////////////////////////////////////////////////////////////
// GETS ALL THE INCLUDE FILES FROM OPENCASCADE 'inc' DIR
Array<String> GetIncludeNames(const String &CasRoot)
{
	// GETS INCLUDE FOLDER CONTENTS
	Array<String> IncArray;
	IncArray = GetDirectoryContent(CasRoot + "/inc/*.*", false);
	
	return IncArray;

} // END GetIncludeNames()

////////////////////////////////////////////////////////////////////////////////////////////////
// GET ALL FOLDERS FROM OPENCASCADE 'src' DIR
Array<String> GetSourceFolders(const String &CasRoot)
{
	// GETS SOURCE FOLDER CONTENT, ONLY FOLDERS
	Array<String> SourceArray;
	SourceArray = GetDirectoryContent(CasRoot + "/src/*.*", true);
	
	return SourceArray;

} // END GetSourceFolders()

////////////////////////////////////////////////////////////////////////////////////////////////
// GET ALL FOLDERS FROM OPENCASCADE 'drv' DIR
Array<String> GetDrvFolders(const String &CasRoot)
{
	// GETS SOURCE FOLDER CONTENT, ONLY FOLDERS
	Array<String> DrvArray;
	DrvArray = GetDirectoryContent(CasRoot + "/drv/*.*", true);
	
	return DrvArray;

} // END GetDrvFolders()

////////////////////////////////////////////////////////////////////////////////////////////////
// FROM THE ARRAY OF INCLUDE FILE NAMES, GETS THE ARRAY OF PACKAGE NAMES
Array<String> GetPackageNames(const Array<String> &IncArray)
{
	Array<String> PackageNames;
	String First, Second;
	for(int i = 0 ; i < IncArray.GetCount() ; i++)
	{
		if(!SplitCascadeName(IncArray[i], First, Second))
			continue;
		if(FindIndex(PackageNames, First) < 0)
			PackageNames << First;
	}
	
	return PackageNames;
	
} // END GetPackageNames()

////////////////////////////////////////////////////////////////////////////////////////////////
// FROM DEST PATH GETS TEMPLATE FILE NAMES
Array<String> GetTemplateNames(const String &DestPath)
{
	// GETS ALL TEMPLATE FILE NAMES ON DEST PATH
	// EXTRACT TEMPLATE NAMES FROM FILENAMES
	Array<String> TemplateFileNames = GetDirectoryContent(DestPath + "/*.template", false);
	String TemplateName;
	Array<String> TemplateNames;
	for(int i = 0 ; i < TemplateFileNames.GetCount() ; i++)
	{
		TemplateName = TemplateFileNames[i];
		int p = TemplateName.Find(".");
		if(p == -1)
			continue;
		TemplateNames << (TemplateName.Mid(0, p));
	}
	
	return TemplateNames;
	
} // END GetTemplateNames()

////////////////////////////////////////////////////////////////////////////////////////////////
// READS A TEMPLATE FILE, RETURNING AN ARRAY OF PACKAGE NAMES
Array<String> ReadTemplate(const String &TemplatePath)
{
	Array<String> TemplateLines;
	String Line;
	
	FileIn f(TemplatePath);
	while(!f.IsEof())
	{
		Line = f.GetLine();
		if(Line == "" || isspace(Line[0]))
			continue;
		TemplateLines << Line;
	}
	
	return TemplateLines;
	
} // END ReadTemplate()

/*
////////////////////////////////////////////////////////////////////////////////////////////////
// THIS FUNCTIONS ADDS A #define TO AN OPENED FILE
void AddDefine(FileOut &fOut, String const &Name, String const &Value)
{
	fOut.PutLine("#ifndef " + Name);
	fOut.PutLine("#define " + Name + " " + Value);
	fOut.PutLine("#endif");
	             
} // END AddDefine()
*/

////////////////////////////////////////////////////////////////////////////////////////////////
// COPIES A FILE PATCHING #INCLUDE DIRECTIVES INSIDE
// IT DOES A LIMITATE CHECK ON FILE, SO IT' NOT ERROR-FREE
// IF SOME STRANGE CODE IS USED
bool CopyPatch(const String &CasRoot, const String &Source, const String &Dest, const Index<String> &Packages, const Array<String> &Categories)
{
	FileIn fIn(Source);
	FileOut fOut(Dest);
	String Line;
	char Delimiter;
	int IncludeStart, IncludeLen;
	String IncludeName;
	String First, Second;
	int iPackage;
	
	enum { none, preproc, inString, inComment, end } State;
	
	State = none;
	while(!fIn.IsEof())
	{
		Line = fIn.GetLine();
		
		int i = 0 ;
		while(i < Line.GetCount())
		{
			switch(State)
			{
				// NORMAL STATE, LOOKS FOR ALL
				case none :
					if(Line[i] == '#')
						State = preproc;
					else if(Line[i] == '"')
						State = inString;
					else if(Line[i] == 0x5c)
						i++;
					else if(Line[i] == '/' && i < Line.GetCount()-1)
					{
						// C++ STYLE COMMENT ?
						if(Line[i+1] == '/')
						{
							// YES, SKIP REST OF LINE...
							i = Line.GetCount();
						}
						// C STYLE COMMENT ?
						else if(Line[i+1] == '*')
						{
							// YES, CHANGE STATE TO inComment
							// AND GO LOOK FOR END OF COMMENT
							State = inComment;
							i++;
						}
					}
					break;

				// INSIDE STRING STATE, LOOKS ONLY FOR END OF STRING
				// OR END OF LINE, WHCH TERMINATES THE STRING TOO
				case inString :
					if(Line[i] == '"' || i >= Line.GetCount()-1)
						State = none;
					else if(Line[i] == 0x5c)
						i++;
					break;
					
				// INSIDE C-STYLE COMMENT, LOOKS ONLY FOR END OF COMMENT
				case inComment :
					if(i < Line.GetCount() -1 && Line[i] == '*' && Line[i+1] == '/')
					{
						State = none;
						i++;
					}
					break;
					
				// PREPROCESSOR DIRECTIVE FOUND, LOOK IF IT'S #include
				case preproc :

					// RESETS STATE FOR FOLLOWING OPERATIONE
					State = none;
					
					// SKIP BLANKS....
					while(i < Line.GetCount() && isspace(Line[i]))
						i++;
					
					if(i < Line.GetCount() - 7 && Line.Mid(i, 7) == "include")
					{
						// GO TO THE END OF DIRECTIVE
						i += 7;
						
						// SKIP BLANKS
						while(i < Line.GetCount() && isspace(Line[i]))
							i++;
						
						// IF END OF LINE, ERROR IN DIRECTIVE, SO SKIP IT
						if(i >= Line.GetCount()-2) // 2 is space for 2 delimiters
						{
							i = Line.GetCount();
							continue;
						}
						
						// GETS INCLUDE DELIMITER AND START POS IF INCLUDE NAME
						Delimiter = Line[i++];
						if(Delimiter == '<')
							Delimiter = '>';
						IncludeStart = i;
						
						// LOOKS FORWARD FOR CLOSING DELIMITER
						while(i < Line.GetCount() && Line[i] != Delimiter)
							i++;
						
						// IF END OF LINE, ERROR IN DIRECTIVE, SO SKIP IT
						if(i >= Line.GetCount())
							continue;
						
						// SAVES INCLUDE NAME LENGTH
						IncludeLen = i - IncludeStart;
						
						// LOOKS IF INCLUDE FILE IS AN OPENCASCADE PACKAGE
						IncludeName = Line.Mid(IncludeStart, IncludeLen);
						if(!SplitCascadeName(IncludeName, First, Second))
							continue;
						if((iPackage = Packages.Find(First)) >= 0)
						{
							// SHOULD BE AN OPENCASCADE ONE
							// LET'S CHECK IF FILE EXISTS... AND WHERE IS IT
							// AND BUILD CORRECT DEST PATH
							if(FileExists(CasRoot + "/inc/" + IncludeName))
								IncludeName = Categories[iPackage] + "/" + Packages[iPackage] + "/" + Second;
							else if(FileExists(CasRoot + "/drv/" + Packages[iPackage] + "/"  + IncludeName))
								IncludeName = Categories[iPackage] + "/" + Packages[iPackage] + "/drv/" + Second;
							else if(FileExists(CasRoot + "/src/" + Packages[iPackage] + "/"  + IncludeName))
								IncludeName = Categories[iPackage] + "/" + Packages[iPackage] + "/src/" + Second;
							else
								continue;
							
							// SUBSTITUTE THE CORRECT INCLUDE IN SOURCE LINE
							Line = Line.Mid(0, IncludeStart-1) + "<" + IncludeName + ">" + Line.Mid(IncludeStart+IncludeLen+2);
							
							// ADJUST THE LINE INDEX
							i = IncludeStart + IncludeName.GetCount() + 1;
						}
					} // if #include
					
					// HERE a DIRTY TRICK TO DEAL WITH #includes REFERRING TO #defined FILES..
					// IT REPLaCES ONLY THE VALUE OF THE MACRO IF AND ONLY IF IT' S SURROUNDED
					// BY <> AND IS PART OF OpenCascade THREE
					// DON'T DEAL WITH INCLUDES WITH '"' DELIMITER... FOR NOW
					if(i < Line.GetCount() - 6 && Line.Mid(i, 6) == "define")
					{
						// GO TO THE END OF DIRECTIVE
						i += 7;
						
						// SKIP BLANKS
						while(i < Line.GetCount() && isspace(Line[i]))
							i++;
						
						// SEARCH FOR < DELIMITER
						while(i < Line.GetCount() && Line[i] != '<')
							i++;
						
						// IF END OF LINE, NOT FOUND, SO SKIP IT
						if(i >= Line.GetCount())
							continue;
						
						// GETS START OF INCLUDE NAME
						i++;
						IncludeStart = i;
						
						// LOOKS FORWARD FOR CLOSING DELIMITER
						while(i < Line.GetCount() && Line[i] != '>')
							i++;
						
						// IF END OF LINE, ERROR IN DIRECTIVE, SO SKIP IT
						if(i >= Line.GetCount())
							continue;
						
						// SAVES INCLUDE NAME LENGTH
						IncludeLen = i - IncludeStart;
						
						// LOOKS IF INCLUDE FILE IS AN OPENCASCADE PACKAGE
						IncludeName = Line.Mid(IncludeStart, IncludeLen);
						if(!SplitCascadeName(IncludeName, First, Second))
							continue;
						if((iPackage = Packages.Find(First)) >= 0)
						{
							// SHOULD BE AN OPENCASCADE ONE
							// LET'S CHECK IF FILE EXISTS... AND WHERE IS IT
							// AND BUILD CORRECT DEST PATH
							if(FileExists(CasRoot + "/inc/" + IncludeName))
								IncludeName = Categories[iPackage] + "/" + Packages[iPackage] + "/" + Second;
							else if(FileExists(CasRoot + "/drv/" + Packages[iPackage] + "/"  + IncludeName))
								IncludeName = Categories[iPackage] + "/" + Packages[iPackage] + "/drv/" + Second;
							else if(FileExists(CasRoot + "/src/" + Packages[iPackage] + "/"  + IncludeName))
								IncludeName = Categories[iPackage] + "/" + Packages[iPackage] + "/src/" + Second;
							else
								continue;

							// SUBSTITUTE THE CORRECT INCLUDE IN SOURCE LINE
							Line = Line.Mid(0, IncludeStart-1) + "<" + IncludeName + ">" + Line.Mid(IncludeStart+IncludeLen+2);
							
							// ADJUST THE LINE INDEX
							i = IncludeStart + IncludeName.GetCount() + 1;
						}
					} // if #define
							
					break;
					
			} // switch
			i++;
		}

		fOut.PutLine(Line);
	}
	return true;

} // END CopyPatch()

////////////////////////////////////////////////////////////////////////////////////////////////
// SCANS A FILE FOR USED PACKAGES
// (CHECK FOR #includes <PACKAGE>
bool GetUsedPackagesFromFile(Index<String> &UsedPackages, Index<String> const &Packages, String const &thisPackage, String const &FilePath)
{
	FileIn fIn(FilePath);
	String Line;
	char Delimiter;
	int IncludeStart, IncludeLen;
	String IncludeName;
	String First, Second;
	int iPackage;
	
	enum { none, preproc, inString, inComment, end } State;
	
	State = none;
	while(!fIn.IsEof())
	{
		Line = fIn.GetLine();
		
		int i = 0 ;
		while(i < Line.GetCount())
		{
			switch(State)
			{
				// NORMAL STATE, LOOKS FOR ALL
				case none :
					if(Line[i] == '#')
						State = preproc;
					else if(Line[i] == '"')
						State = inString;
					else if(Line[i] == 0x5c)
						i++;
					else if(Line[i] == '/' && i < Line.GetCount()-1)
					{
						// C++ STYLE COMMENT ?
						if(Line[i+1] == '/')
						{
							// YES, SKIP REST OF LINE...
							i = Line.GetCount();
						}
						// C STYLE COMMENT ?
						else if(Line[i+1] == '*')
						{
							// YES, CHANGE STATE TO inComment
							// AND GO LOOK FOR END OF COMMENT
							State = inComment;
							i++;
						}
					}
					break;

				// INSIDE STRING STATE, LOOKS ONLY FOR END OF STRING
				// OR END OF LINE, WHCH TERMINATES THE STRING TOO
				case inString :
					if(Line[i] == '"' || i >= Line.GetCount()-1)
						State = none;
					else if(Line[i] == 0x5c)
						i++;
					break;
					
				// INSIDE C-STYLE COMMENT, LOOKS ONLY FOR END OF COMMENT
				case inComment :
					if(i < Line.GetCount() -1 && Line[i] == '*' && Line[i+1] == '/')
					{
						State = none;
						i++;
					}
					break;
					
				// PREPROCESSOR DIRECTIVE FOUND, LOOK IF IT'S #include
				case preproc :

					// RESETS STATE FOR FOLLOWING OPERATIONE
					State = none;
					
					// SKIP BLANKS....
					while(i < Line.GetCount() && isspace(Line[i]))
						i++;
					
					if(i < Line.GetCount() - 7 && Line.Mid(i, 7) == "include")
					{
						// GO TO THE END OF DIRECTIVE
						i += 7;
						
						// SKIP BLANKS
						while(i < Line.GetCount() && isspace(Line[i]))
							i++;
						
						// IF END OF LINE, ERROR IN DIRECTIVE, SO SKIP IT
						if(i >= Line.GetCount()-2) // 2 is space for 2 delimiters
						{
							i = Line.GetCount();
							continue;
						}
						
						// GETS INCLUDE DELIMITER AND START POS IF INCLUDE NAME
						Delimiter = Line[i++];
						if(Delimiter == '<')
							Delimiter = '>';
						IncludeStart = i;
						
						// LOOKS FORWARD FOR CLOSING DELIMITER
						while(i < Line.GetCount() && Line[i] != Delimiter)
							i++;
						
						// IF END OF LINE, ERROR IN DIRECTIVE, SO SKIP IT
						if(i >= Line.GetCount())
							continue;
						
						// SAVES INCLUDE NAME LENGTH
						IncludeLen = i - IncludeStart;
						
						// GETS INCLUDE FILE NAME
						IncludeName = Line.Mid(IncludeStart, IncludeLen);
						
						// GETS PACKAGE NAME FROM INCLUDE NAME (IF ANY...)
						int iLastSlash = IncludeName.ReverseFind('/');
						if(iLastSlash > 0)
						{
							int iPrevSlash = IncludeName.ReverseFind('/', iLastSlash-1);
							if(iPrevSlash != -1 && iLastSlash - iPrevSlash > 1)
							{
								String PackageName = IncludeName.Mid(iPrevSlash+1, iLastSlash-iPrevSlash-1);
														
								// LOOKS IF INCLUDE FILE IS AN OPENCASCADE PACKAGE
								// AND IS NOT THE PACKAGE CURRENTLY SCANNED
								if(PackageName != thisPackage && Packages.Find(PackageName) >= 0 && UsedPackages.Find(PackageName) < 0)
								{
									// YES, ADD IT TO USED PACKAGES LIST
									UsedPackages << PackageName;
								}
							}
						}
					}
							
					break;
					
			} // switch
			i++;
		}

	}
	return true;

} // END GetUsedPackagesFromFile()

////////////////////////////////////////////////////////////////////////////////////////////////
// ADDS FILE OPTIONS TO PACKAGE DESCRIPTION FILE
void AddFileOptions(FileOut &upp, String const &FileName)
{
	String Ext = GetFileExt(FileName);
	if( (Ext == ".cpp" || Ext == ".cxx") && (String(CXX_OPTIONS) != ""))
	{
		upp.PutLine("");
		upp.Put("\t\toptions() \"" + String(CXX_OPTIONS) + "\"");
	}
	else if( Ext == ".c" && String(C_OPTIONS) != "")
	{
		upp.PutLine("");
		upp.Put("\t\toptions() \"" + String(C_OPTIONS) + "\"");
	}
} // END AddFileOptions()

////////////////////////////////////////////////////////////////////////////////////////////////
// PACKS OPENCASCADE FOR UPP USAGE
bool PackageCascade(const String &CasRoot, const String &DestPath)
{
	// GETS THE TEMPLATE NAMES FROM DESTINATION DIRECTORY
	// (MUST EXIST AND FILLED WITH INCLUDE FILE NAMES)
	Array<String> TemplateNames = GetTemplateNames(DestPath);

	// FOR EACH TEMPLATE, EXTRACTS INCLUDED PACKAGE NAMES
	// GETTING THEM FROM INCLUDE FILE NAMES AND MAPS THEM TO TEMPLATE NAME
	Index<String> Packages;
	Array<String> Categories;
	for(int i = 0 ; i < TemplateNames.GetCount() ; i++)
	{
		// READS THE TEMPLATE FILE
		Array<String> TemplateLines = ReadTemplate(DestPath + "/" + TemplateNames[i] + ".template");
		
		// EXTRACTS THE PACKAGE NAMES
		Array<String> TemplatePackages = GetPackageNames(TemplateLines);
		
		// MAPS THE PACKAGE NAMES WITH CORRESPONDING TEMPLATE
		// AND FILLS THE GLOBAL PACKAGES ARRAY FROM TEMPLATES
		for(int j = 0 ; j < TemplatePackages.GetCount() ; j++)
		{
			Packages.Add(TemplatePackages[j]);
			Categories.Add(TemplateNames[i]);
		}
	}
	
	// HERE WE MUST DEAL WITH INCLUDE FILES THAT ARE NOT INCLUDED IN PACKAGES
	// INSIDE THE TEMPLATES; SHOULD NOT HAPPEN, BUT... WE MAP THEM TO 'Others' TEMPLATE
	
	// FIRST, WE READ THE INCLUDE FILES....
	Array<String> IncludeFiles = GetIncludeNames(CasRoot);
	
	// ...FROM THEM WE EXTRACT A PACKAGE NAMES LIST....
	Array<String> PackagesFromIncludes = GetPackageNames(IncludeFiles);
	
	// AND WE LOOK FOR PACKAGES NOT INCLUDED IN TEMPLATES
	// MAPPING THEM TO 'Others' CATEGORY
	bool FoundOrphan = false;
	for(int i = 0 ; i < PackagesFromIncludes.GetCount() ; i++)
	{
		if(Packages.Find(PackagesFromIncludes[i]) < 0)
		{
			Packages.Add(PackagesFromIncludes[i]);
			Categories.Add("Others");
			FoundOrphan = true;
		}
	}
	if(FoundOrphan)
		TemplateNames << "Others";
	
	////////////////////////////////////////////////////////////////////////////////////////////
	// HERE WE HAVE :
	//  All template (category) names in TemplateNames
	//  2 Arrays mapping  Package names with corresponding Templates in Packages+Categories
	//  the complete array of include file names in IncludeFiles
	//
	// We must now create package folders on destination path
	// and copy include files there, changing #include directives if necessary
	
	// FIRST, WE CREATE THE CATEGORY PATHS...
	String ErrorStr;
	String FullPath;
	String SrcPath, DrvPath, OthersPath;
	for(int i = 0 ; i < TemplateNames.GetCount() ; i++)
	{
		FullPath = DestPath + "/" + TemplateNames[i];
		if(!StdFileSystemInfo().FolderExists(FullPath))
		{
			if(!StdFileSystemInfo().CreateFolder(FullPath, ErrorStr))
			{
				Cout() << "Error '" << ErrorStr << " creating '" << FullPath << "'\n";
				return false;
			}
		}
	}
	
	// THEN THE PACKAGES PATHS.....
	for(int i = 0 ; i < Packages.GetCount() ; i++)
	{
		FullPath = DestPath + "/" + Categories[i] + "/" + Packages[i];
		if(!StdFileSystemInfo().FolderExists(FullPath))
		{
			if(!StdFileSystemInfo().CreateFolder(FullPath, ErrorStr))
			{
				Cout() << "Error '" << ErrorStr << " creating '" << FullPath << "'\n";
				return false;
			}
		}
	}
	
	// NOW WE MUST COPY ALL INCLUDE FILES, PATCHING THE INCLUDE DIRECTIVES
	for(int i = 0 ; i < IncludeFiles.GetCount() ; i++)
	{
		String First, Second;
		if(!SplitCascadeName(IncludeFiles[i], First, Second))
		  continue;
		String Source = CasRoot + "/inc/" + IncludeFiles[i];
		int iPackage = Packages.Find(First);
		String Category = Categories[iPackage];
		String Dest = DestPath + "/" + Category + "/" + First + "/" + Second;
		CopyPatch(CasRoot, Source, Dest, Packages, Categories);
	}
	
	// WE CREATE A FOLDER FOR SOURCES OUTSIDE PACKAGES
	// (THERE ARE SOME....)
	FullPath = DestPath + "/_UNKNOWN_";
	if(!StdFileSystemInfo().CreateFolder(FullPath, ErrorStr))
	{
		Cout() << "Error '" << ErrorStr << " creating '" << FullPath << "'\n";
		return false;
	}
	
	//////////////////////////////////////////////////////////////////////////////////////////////
	// NOW WE MUST DEAL WITH SOURCE FILES; WE PUT THEM IN src OR others SUBFOLDER OF EACH PACKAGE
	// IN ORDER TO NOT CLOBBER TOO MUCH THE DIRECTORIES
	
	// AT FIRST, WE GET A LIST OF SOURCE FILE FOLDERS....
	Array<String> SourceFolders = GetSourceFolders(CasRoot);
	
	// NOW WE PROCESS EACH SOURCE FOLDER
	for(int i = 0 ; i < SourceFolders.GetCount() ; i++)
	{
		// IF SOURCE FOLDER NAME IS A PACKAGE NAME (AS IT SHOULD)
		// WE DON'T NEED TO CREATE IT ON DEST; WE MUST ONLY BUILD
		// DEST PATH
		int iPackage;
		if( (iPackage = Packages.Find(SourceFolders[i])) >= 0)
		{
		   // OK, CREATE DEST PATH ONLY
			FullPath = DestPath + "/" + Categories[iPackage] + "/" + Packages[iPackage];
		}
		else
		{
			// MMMHHH.. SOURCE DIR DOESN'T BELONG TO A PACKAGE
			// CREATE DEST PATH ON _UNKNOWN_ DEST PATH
			FullPath = DestPath + "/_UNKNOWN_/" + SourceFolders[i];
			if(!StdFileSystemInfo().CreateFolder(FullPath, ErrorStr))
			{
				Cout() << "Error '" << ErrorStr << " creating '" << FullPath << "'\n";
				return false;
			}

		}
		SrcPath = FullPath + "/src";
		OthersPath = FullPath + "/others";
		
		// NOW WE CREATE 'src' SUBFOLDER
		if(!StdFileSystemInfo().CreateFolder(SrcPath, ErrorStr))
		{
			Cout() << "Error '" << ErrorStr << " creating '" << SrcPath << "'\n";
			return false;
		}
		
		// NOW WE CREATE 'others' SUBFOLDER
		if(!StdFileSystemInfo().CreateFolder(OthersPath, ErrorStr))
		{
			Cout() << "Error '" << ErrorStr << " creating '" << OthersPath << "'\n";
			return false;
		}
		
		// WE'RE DONE WITH FOLDERS, NOW WE DEAL WITH FILES ON EACH FOLDER
		Array<String> SourceFiles = GetDirectoryContent(CasRoot + "/src/" + SourceFolders[i] + "/*.*", false);
		Index<String> SourceExt;
		SourceExt << ".c" << ".cxx" << ".cpp" << ".h" << ".hxx" << ".hpp" << ".lxx" << ".gxx" << ".pxx";
		for(int j = 0 ; j < SourceFiles.GetCount() ; j++)
		{
			// WE GET THE FILENAME
			String SourceFile = SourceFiles[j];
			
			// AND WE SPLIT INTO PACKAGE NAME AND REAL FILE NAME
			String First, Second;
			if(!SplitCascadeName(SourceFile, First, Second))
				continue;
			
			// FIRST WE CHECK THE FILE EXTENSION
			// AND WE CREATE SOURCE AND DESTINATION FULL PATH
			String Source = CasRoot + "/src/" + SourceFolders[i] + "/" + SourceFile;
			String Dest;
			if(SourceExt.Find(GetFileExt(SourceFile)) >= 0)
				Dest = SrcPath + "/" + Second;
			else
				Dest = OthersPath + "/" + Second;
			
			// AND WE COPY-PATCH THE FILE
			CopyPatch(CasRoot, Source, Dest, Packages, Categories);
		}
	}
	
	//////////////////////////////////////////////////////////////////////////////////////////////
	// NOW WE MUST DEAL WITH DRV FILES; WE PUT THEM IN drv SUBFOLDER OF EACH PACKAGE
	// IN ORDER TO NOT CLOBBER TOO MUCH THE DIRECTORIES
	
	// AT FIRST, WE GET A LIST OF DRV FILE FOLDERS....
	Array<String> DrvFolders = GetDrvFolders(CasRoot);
	
	// NOW WE PROCESS EACH DRV FOLDER
	for(int i = 0 ; i < DrvFolders.GetCount() ; i++)
	{
		// IF DRV FOLDER NAME IS A PACKAGE NAME (AS IT SHOULD)
		// WE DON'T NEED TO CREATE IT ON DEST; WE MUST ONLY BUILD
		// DEST PATH
		int iPackage;
		if( (iPackage = Packages.Find(DrvFolders[i])) >= 0)
		{
		   // OK, CREATE DEST PATH ONLY
			FullPath = DestPath + "/" + Categories[iPackage] + "/" + Packages[iPackage];
		}
		else
		{
			// MMMHHH.. DRV DIR DOESN'T BELONG TO A PACKAGE
			// CREATE DEST PATH ON _UNKNOWN_ DEST PATH, IF NOT ALREADY DONE BY SOURCE FILES PROCESS
			FullPath = DestPath + "/_UNKNOWN_/" + DrvFolders[i];
			if(!DirectoryExists(FullPath))
			{
				if(!StdFileSystemInfo().CreateFolder(FullPath, ErrorStr))
				{
					Cout() << "Error '" << ErrorStr << " creating '" << FullPath << "'\n";
					return false;
				}
			}

		}
		DrvPath = FullPath + "/drv";
		
		// NOW WE CREATE 'drv' SUBFOLDER
		if(!StdFileSystemInfo().CreateFolder(DrvPath, ErrorStr))
		{
			Cout() << "Error '" << ErrorStr << " creating '" << DrvPath << "'\n";
			return false;
		}
		
		// WE'RE DONE WITH FOLDERS, NOW WE DEAL WITH FILES ON EACH FOLDER
		Array<String> DrvFiles = GetDirectoryContent(CasRoot + "/drv/" + DrvFolders[i] + "/*.*", false);
		for(int j = 0 ; j < DrvFiles.GetCount() ; j++)
		{
			// WE GET THE FILENAME
			String DrvFile = DrvFiles[j];
			
			// AND WE SPLIT INTO PACKAGE NAME AND REAL FILE NAME
			String First, Second;
			if(!SplitCascadeName(DrvFile, First, Second))
				continue;
			
			// FIRST WE CHECK THE FILE EXTENSION
			// AND WE CREATE SOURCE AND DESTINATION FULL PATH
			String Source = CasRoot + "/drv/" + DrvFolders[i] + "/" + DrvFile;
			String Dest = DrvPath + "/" + Second;
			
			// AND WE COPY-PATCH THE FILE
			CopyPatch(CasRoot, Source, Dest, Packages, Categories);
		}
	}
	
	/////////////////////////////////////////////////////////////////////////////////////////
	// NOW WE MUST ONLY CREATE THE PACKAGE DESCRIPTION FILES
	// WE SCAN AGAIN EACH PACKAGE, GATHERING ALL FILES INSIDE IT
	for(int iTemplate = 0 ; iTemplate < TemplateNames.GetCount() ; iTemplate++)
	{
		// GETS CURRENT CATEGORY NAME
		String Category = TemplateNames[iTemplate];
		
		// SCANS FOR PACKAGES IN THAT CATEGORY
		Array<String> CategoryPackages = GetDirectoryContent(DestPath + "/" + Category + "/*.*", true);
		
		// FOR EACH PACKAGE IN CURRENT CATEGORY
		for(int iPackage = 0 ; iPackage < CategoryPackages.GetCount() ; iPackage++)
		{
			String PackageName = CategoryPackages[iPackage];
			String PackagePath = DestPath + "/" + Category + "/" + PackageName;
			
			// GETS ALL INCLUDE FILE NAMES FROM PACKAGE
			Array<String> IncludeFiles = GetDirectoryContent(PackagePath + "/*.*", false);
			Sort(IncludeFiles);
			
			// GETS ALL SOURCE FILE NAMES FROM PACKAGE
			Array<String> SourceFiles = GetDirectoryContent(PackagePath + "/src/*.*", false);
			Sort(SourceFiles);
			
			// GETS ALL DRV FILE NAMES FROM PACKAGE
			Array<String> DrvFiles = GetDirectoryContent(PackagePath + "/drv/*.*", false);
			Sort(DrvFiles);
			
			// GETS ALL OTHER FILE NAMES FROM PACKAGE
			Array<String> OthersFiles = GetDirectoryContent(PackagePath + "/others/*.*", false);
			Sort(OthersFiles);
			
			// CREATES LIST OF USED PACKAGES
			Index<String> UsedPackages;
			
			// GATHER USED PACKAGES FROM ALL SOURCE, DRV AND INCLUDE FILES IN PACKAGE DIRECTORY
			// SKIPS FILES IN others DIR, THEY SHOULD NOT BE SOURCES....
			for(int iFile = 0 ; iFile < IncludeFiles.GetCount() ; iFile++)
				GetUsedPackagesFromFile(UsedPackages, Packages, PackageName, PackagePath + "/" + IncludeFiles[iFile]);
			for(int iFile = 0 ; iFile < SourceFiles.GetCount() ; iFile++)
				GetUsedPackagesFromFile(UsedPackages, Packages, PackageName, PackagePath + "/src/" + SourceFiles[iFile]);
			for(int iFile = 0 ; iFile < DrvFiles.GetCount() ; iFile++)
				GetUsedPackagesFromFile(UsedPackages, Packages, PackageName, PackagePath + "/drv/" + DrvFiles[iFile]);
			
			// CREATES .upp FILE FROM DATA
			FileOut upp(PackagePath + "/" + PackageName + ".upp");
			
			// DESCRIPTION
			upp.PutLine("description \"Package " + PackageName + "\";");
			upp.PutLine("");
			
			// USED PACKAGES
			if(UsedPackages.GetCount())
			{
				upp.PutLine("uses");
				String UsedPackage = UsedPackages[0];
				int j = Packages.Find(UsedPackage);
				upp.Put("\t" + Categories[j] + "/" + UsedPackage);
				for(int i = 1 ; i < UsedPackages.GetCount() ; i++)
				{
					upp.PutLine(",");
					UsedPackage = UsedPackages[i];
					j = Packages.Find(UsedPackage);
					upp.Put("\t" + Categories[j] + "/" + UsedPackage);
				}
				upp.PutLine(";");
				upp.PutLine("");
			}
			
			// PACKAGE OPTIONS
			if(String(PACKAGE_OPTIONS) != "")
			{
				upp.PutLine("options");
				upp.PutLine(String("\t\"") + PACKAGE_OPTIONS + "\";");
				upp.PutLine("");
			}
			
			// FILES IN PACKAGE
			// FILLS LIST FROM PACKAGE FILES
			if(IncludeFiles.GetCount() || SourceFiles.GetCount() || DrvFiles.GetCount() || OthersFiles.GetCount())
			{
				upp.PutLine("file");
				if(IncludeFiles.GetCount())
				{
					upp.Put("\tHeaders readonly separator");
					for(int iFile = 0 ; iFile < IncludeFiles.GetCount() ; iFile++)
					{
						upp.PutLine(",");
						upp.Put("\t" + IncludeFiles[iFile]);
					}
				}
				if(SourceFiles.GetCount())
				{
					if(IncludeFiles.GetCount())
						upp.PutLine(",");
					upp.Put("\tSources readonly separator");
					for(int iFile = 0 ; iFile < SourceFiles.GetCount() ; iFile++)
					{
						upp.PutLine(",");
						upp.Put("\tsrc/" + SourceFiles[iFile]);
						AddFileOptions(upp, SourceFiles[iFile]);
					}
				}
				if(DrvFiles.GetCount())
				{
					if(IncludeFiles.GetCount() || SourceFiles.GetCount())
						upp.PutLine(",");
					upp.Put("\tDrv readonly separator");
					for(int iFile = 0 ; iFile < DrvFiles.GetCount() ; iFile++)
					{
						upp.PutLine(",");
						upp.Put("\tdrv/" + DrvFiles[iFile]);
						AddFileOptions(upp, DrvFiles[iFile]);
					}
				}
				if(OthersFiles.GetCount())
				{
					if(IncludeFiles.GetCount() || SourceFiles.GetCount() || DrvFiles.GetCount())
						upp.PutLine(",");
					upp.Put("\tOthers readonly separator");
					for(int iFile = 0 ; iFile < OthersFiles.GetCount() ; iFile++)
					{
						upp.PutLine(",");
						upp.Put("\tothers/" + OthersFiles[iFile]);
					}
				}
			
				upp.PutLine(";");
			}
			upp.PutLine("");
/*
			// MAINCONFIG SECTION
			upp.PutLine("mainconfig");
			upp.PutLine("\t\"\" = \"\";");
*/
		} // for iPackage
	} // for iCategory

	return true;

} // END PackageCascade()

////////////////////////////////////////////////////////////////////////////////////////////////
// CREATES THE TOP LEVEL PACKAGES, GROUPING ALL PACKAGES FOR EACH CATHEGORY
// MAKES ALSO GLOBAL INCLUDE FILES, IF ONE IS TOO LAZY TO INCLUDE SINGLE PACKAGES HEADERS...
bool BuildToplevelPackages(const String &DestPath)
{
	// GETS PACKED OPENCASCADE DIRECTORY CONTENTS
	Array<String> Categories = GetDirectoryContent(DestPath + "/*.*", true);
	
	// TAKES OFF _UNKNOWN_ PACKAGE, IT'S NOT NEEDED
	int k = FindIndex(Categories, "_UNKNOWN_");
	if(k >= 0)
		Categories.Remove(k);
	
	// LOOP FOR EACH CATEGORY
	for(int iCategory = 0 ; iCategory < Categories.GetCount() ; iCategory++)
	{
		// GETS THE CURRENT CATEGORY NAME AND PATH
		String Category = Categories[iCategory];
		String CategoryPath = DestPath + "/" + Category;
		
		// GETS PACKAGE LIST FROM CATEGORY
		Array<String> Packages = GetDirectoryContent(CategoryPath + "/*.*", true);
		
		// CREATES PACKAGE DESCRIPTION FILE
		FileOut upp(CategoryPath + "/" + Category + ".upp");
		upp.PutLine("description \"Package " + Category + "\";");
		upp.PutLine("");
		
		// FILLS USED PACKAGES
		if(Packages.GetCount())
		{
			upp.PutLine("uses");
			upp.Put("\t" + Category + "/" + Packages[0]);
			for(int iPackage = 1 ; iPackage < Packages.GetCount() ; iPackage++)
			{
				upp.PutLine(",");
				upp.Put("\t" + Category + "/" + Packages[iPackage]);
			}
			upp.PutLine(";");
			upp.PutLine("");
		}
		
		// FILLS PACKAGE OPTIONS
		if(String(PACKAGE_OPTIONS) != "")
		{
			upp.PutLine("options");
			upp.PutLine(String("\t\"") + PACKAGE_OPTIONS + "\";");
			upp.PutLine("");
		}
			
/*		
		// FILLS FILES IN PACKAGE
		// (JUST THE MAIN INCLUDE FILE)
		upp.PutLine("file");
		upp.PutLine("\t" + Category + ".hxx");

		// CREATES MAIN PACKAGE INCLUDE FILE
		FileOut inc(CategoryPath + "/" + Category + ".hxx");
		
		// SCANS FILES IN EACH USED PACKAGE AND FILLS MAIN INCLUDE FILE
		for(int iPackage = 0 ; iPackage < Packages.GetCount() ; iPackage++)
		{
			String Package = Packages[iPackage];
			inc.PutLine("// Includes from package '" + Package + "'");
			Array<String> Includes = GetDirectoryContent(CategoryPath + "/" + Package + "/*.hxx", false);
			for(int iInclude = 0 ; iInclude < Includes.GetCount(); iInclude++)
				inc.PutLine("#include <" + CategoryPath + "/" + Package + "/" + Includes[iInclude] + ">");
			inc.PutLine("");
		} // for iPackage
*/
		
	} // for iCategory
} // END BuildToplevelPackages()

CONSOLE_APP_MAIN
{
	int argc = CommandLine().GetCount();
	const Vector<String>& argv = CommandLine();
	String CasRoot;

	if(argc == 1)
	{
		// CHECKS FOR DESTINATION FOLDER
		if(!FileSystemInfo().FolderExists(argv[0]))
		{
			Cout() << "Destination folder '" << argv[0] << "' don't exist";
			exit(1);
		}
		
		// CHECKS FOR EXISTING OPENCASCADE
		CasRoot = GetCasRoot();
		if(CasRoot == "")
		{
			Cout() << "OpenCASCADE folder not found\nPlease check OpenCASCADE setup\n";
			exit(1);
		}

		PackageCascade(CasRoot, argv[0]);
		BuildToplevelPackages(argv[0]);
		SetExitCode(0);
	}
	else
	{
		Cout() << "USAGE : PackCascade <dest root path>\n";
		exit(1);
	}
}
